import fs from 'fs/promises';
import path from 'path';
import { fileURLToPath } from 'url';
import { getPool } from './pool.js';
import { logger } from '../utils/logger.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

async function runMigrations(): Promise<void> {
  console.log('🔄 Starting migration process...');
  const pool = await getPool();
  const migrationsDir = path.join(__dirname, 'migrations');

  try {
    console.log('📂 Looking for migrations in:', migrationsDir);
    // Read all migration files
    const files = await fs.readdir(migrationsDir);
    const sqlFiles = files
      .filter(f => f.endsWith('.sql'))
      .sort(); // Execute in alphabetical order

    if (sqlFiles.length === 0) {
      console.log('⚠️  No migration files found');
      logger.warn('No migration files found');
      return;
    }

    console.log(`✓ Found ${sqlFiles.length} migration file(s)`);
    logger.info(`Found ${sqlFiles.length} migration file(s)`);

    // Execute each migration
    for (const file of sqlFiles) {
      const filePath = path.join(migrationsDir, file);
      const sql = await fs.readFile(filePath, 'utf-8');
      
      console.log(`🔄 Running migration: ${file}`);
      logger.info(`Running migration: ${file}`);
      
      // Split by semicolon and execute each statement
      const statements = sql
        .split(';')
        .map(s => s.trim())
        .filter(s => s.length > 0);

      for (const statement of statements) {
        await pool.query(statement);
      }
      
      console.log(`✓ Completed: ${file}`);
      logger.info(`✓ Completed: ${file}`);
    }

    console.log('✅ All migrations completed successfully!');
    logger.info('✓ All migrations completed successfully');
  } catch (error) {
    console.error('❌ Migration failed:', error);
    logger.error({ err: error }, 'Migration failed');
    throw error;
  } finally {
    await pool.end();
  }
}

// Run migrations if this file is executed directly
const isMainModule = import.meta.url.endsWith('migrate.ts') || import.meta.url.endsWith('migrate.js');
if (isMainModule) {
  console.log('Starting migrations...');
  runMigrations()
    .then(() => {
      console.log('✅ Migration process completed successfully');
      logger.info('Migration process completed');
      process.exit(0);
    })
    .catch((error) => {
      console.error('❌ Migration process failed:', error);
      logger.error({ err: error }, 'Migration process failed');
      process.exit(1);
    });
}

export { runMigrations };
