import { Request, Response, NextFunction } from 'express';
import { AppError } from './errors.js';

// Role-based permissions map
// Role IDs: 1=Admin, 2=Manager, 3=Clerk, 4=Viewer (from seed data)
const permissions: Record<string, number[]> = {
  // Departments
  'departments:read': [1, 2, 3, 4],
  'departments:create': [1, 2],
  'departments:update': [1, 2],
  'departments:delete': [1],

  // Roles
  'roles:read': [1, 2, 3, 4],
  'roles:create': [1, 2],
  'roles:update': [1, 2],
  'roles:delete': [1],

  // Employees
  'employees:read': [1, 2, 3, 4],
  'employees:create': [1, 2],
  'employees:update': [1, 2],
  'employees:delete': [1],

  // Users
  'users:read': [1, 2],
  'users:create': [1],
  'users:update': [1],
  'users:delete': [1],

  // Dashboard
  'dashboard:read': [1, 2, 3, 4],

  // Matters (Phase 6)
  'matters:read': [1, 2, 3, 4],
  'matters:create': [1, 2],
  'matters:update': [1, 2],
  'matters:delete': [1],

  // Conflicts (Phase 6)
  'conflicts:check': [1, 2, 3],      // Admin, Manager, Clerk can run checks
  'conflicts:override': [1, 2],      // Only Admin, Manager can override

  // Workflows (Phase 6)
  'workflows:read': [1, 2, 3, 4],
  'workflows:manage': [1],           // Only Admin can create/edit workflows
  'workflows:transition': [1, 2, 3], // Admin, Manager, Clerk can transition stages

  // Courts (Phase 7)
  'courts:read': [1, 2, 3, 4],       // All roles can view court calendars
  'courts:create': [1, 2, 3],        // Admin, Manager, Clerk can create calendars
  'courts:update': [1, 2, 3],        // Admin, Manager, Clerk can update calendars
  'courts:delete': [1, 2],           // Admin, Manager can delete

  // Evidence (Phase 7)
  'evidence:read': [1, 2, 3, 4],     // All roles can view evidence
  'evidence:create': [1, 2, 3],      // Admin, Manager, Clerk can create evidence
  'evidence:update': [1, 2, 3],      // Admin, Manager, Clerk can update evidence
  'evidence:delete': [1, 2],         // Admin, Manager can delete
  'evidence:custody': [1, 2, 3],     // Admin, Manager, Clerk can record custody events

  // Time Tracking (Phase 8)
  'time:create': [1, 2, 3, 4],       // All roles can create time entries
  'time:approve': [1, 2],            // Admin, Manager can approve time
  'time:view-all': [1, 2],           // Admin, Manager can view all entries

  // Expense Management (Phase 8)
  'expense:create': [1, 2, 3, 4],    // All roles can create expenses
  'expense:approve': [1, 2],         // Admin, Manager can approve expenses
  'expense:view-all': [1, 2],        // Admin, Manager can view all expenses

  // Invoicing (Phase 8)
  'invoice:read': [1, 2, 3, 4],      // All roles can read invoices
  'invoice:create': [1, 2],          // Admin, Manager can create invoices
  'invoice:update': [1, 2],          // Admin, Manager can update invoices
  'invoice:send': [1, 2],            // Admin, Manager can send invoices
  'invoice:payment': [1, 2],         // Admin, Manager can record payments

  // Receipts & Collections (Phase 8)
  'receipt:read': [1, 2, 3, 4],      // All roles can read receipts
  'receipt:create': [1, 2],          // Admin, Manager can create receipts
  'receipt:update': [1, 2],          // Admin, Manager can update receipts

  // Trust Accounting (Phase 8)
  'trust:read': [1, 2],              // Admin, Manager can read trust accounts
  'trust:manage': [1],               // Admin only can manage accounts
  'trust:transact': [1, 2],          // Admin, Manager can create transactions
  'trust:approve': [1, 2],           // Admin, Manager can approve (dual control)

  // WIP & Revenue Recognition (Phase 8)
  'wip:read': [1, 2],                // Admin, Manager can read WIP and revenue
  'wip:manage': [1, 2],              // Admin, Manager can manage revenue recognition
};

export const requirePermission = (permission: string) => {
  return (req: Request, res: Response, next: NextFunction): void => {
    if (!req.user) {
      throw new AppError(401, 'Authentication required');
    }

    const allowedRoles = permissions[permission] || [];

    if (!allowedRoles.includes(req.user.role_id)) {
      throw new AppError(403, 'Insufficient permissions');
    }

    next();
  };
};

export const requireRole = (...roleIds: number[]) => {
  return (req: Request, res: Response, next: NextFunction): void => {
    if (!req.user) {
      throw new AppError(401, 'Authentication required');
    }

    if (!roleIds.includes(req.user.role_id)) {
      throw new AppError(403, 'Insufficient permissions');
    }

    next();
  };
};
