import { Request, Response, NextFunction } from 'express';
import { ZodSchema, ZodError } from 'zod';

export const validate = (schema: ZodSchema) => {
  return async (req: Request, res: Response, next: NextFunction): Promise<void> => {
    try {
      await schema.parseAsync({
        body: req.body,
        query: req.query,
        params: req.params,
      });
      next();
    } catch (error) {
      if (error instanceof ZodError) {
        const errors: Record<string, string> = {};
        
        error.errors.forEach((err) => {
          const path = err.path.slice(1).join('.'); // Remove 'body'/'query'/'params' prefix
          errors[path] = err.message;
        });

        res.status(422).json({
          success: false,
          error: 'Validation failed',
          errors,
        });
        return;
      }
      next(error);
    }
  };
};
