import { z } from 'zod';

// ====================
// Request Schemas
// ====================

export const runConflictCheckSchema = z.object({
  body: z.object({
    matter_id: z.number().int().positive(),
    search_queries: z.array(z.string().min(2)).min(1, 'At least one search query required'),
    check_client: z.boolean().default(true),
    check_parties: z.boolean().default(true),
    check_counsel: z.boolean().default(true),
  }),
});

export const getConflictCheckSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
});

export const overrideConflictSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
  body: z.object({
    justification: z.string().min(10, 'Justification must be at least 10 characters'),
  }),
});

export const getMatterConflictsSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
});

// ====================
// TypeScript Types
// ====================

export type RunConflictCheckDto = z.infer<typeof runConflictCheckSchema>['body'];
export type OverrideConflictDto = z.infer<typeof overrideConflictSchema>['body'];

export interface ConflictMatch {
  matter_id: number;
  matter_number: string;
  case_type: string;
  party_name: string;
  party_type: string;
  match_type: 'Exact' | 'Fuzzy' | 'Related';
  similarity_score: number;
  identifier_match: boolean;
}

export interface ConflictCheckResult {
  query: string;
  matches: ConflictMatch[];
  conflict_level: 'None' | 'Name Similar' | 'Indirect' | 'Direct';
}

export interface ConflictCheckResponse {
  id: number;
  matter_id: number;
  check_query: string;
  run_by: number;
  run_by_name: string;
  run_at: string;
  result_count: number;
  result_summary: string | null;
  conflict_level: string;
  override_flag: boolean;
  override_by: number | null;
  override_by_name: string | null;
  override_at: string | null;
  override_justification: string | null;
  results: ConflictCheckResult[];
}
