import { z } from 'zod';

// ====================
// Request Schemas
// ====================

export const createCourtCalendarSchema = z.object({
  body: z.object({
    matter_id: z.number().int().positive(),
    hearing_id: z.number().int().positive().optional(),
    court_type_id: z.number().int().positive().optional(),
    court_level: z.enum(['District Court', 'Circuit Court', 'High Court', 'Court of Appeal', 'Supreme Court']),
    court_location: z.string().min(2).max(200),
    court_room: z.string().max(50).optional(),
    judge_name: z.string().max(200).optional(),
    docket_number: z.string().max(100).optional(),
    hearing_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
    hearing_time: z.string().regex(/^\d{2}:\d{2}(:\d{2})?$/).optional(),
    hearing_type: z.enum(['Hearing', 'Motion', 'Trial', 'Ruling', 'Judgment', 'Mention', 'Case Management', 'Other']).default('Hearing'),
    duration_minutes: z.number().int().positive().optional().default(60),
    assigned_lawyer_id: z.number().int().positive(),
    notes: z.string().optional(),
  }),
});

export const updateCourtCalendarSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
  body: z.object({
    court_type_id: z.number().int().positive().optional(),
    court_level: z.enum(['District Court', 'Circuit Court', 'High Court', 'Court of Appeal', 'Supreme Court']).optional(),
    court_location: z.string().min(2).max(200).optional(),
    court_room: z.string().max(50).optional(),
    judge_name: z.string().max(200).optional(),
    docket_number: z.string().max(100).optional(),
    hearing_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    hearing_time: z.string().regex(/^\d{2}:\d{2}(:\d{2})?$/).optional(),
    hearing_type: z.enum(['Hearing', 'Motion', 'Trial', 'Ruling', 'Judgment', 'Mention', 'Case Management', 'Other']).optional(),
    duration_minutes: z.number().int().positive().optional(),
    status: z.enum(['Scheduled', 'Confirmed', 'Postponed', 'Completed', 'Cancelled']).optional(),
    assigned_lawyer_id: z.number().int().positive().optional(),
    notes: z.string().optional(),
    postponement_reason: z.string().optional(),
    outcome: z.string().optional(),
  }),
});

export const getCourtCalendarSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
});

export const listCourtCalendarsSchema = z.object({
  query: z.object({
    page: z.string().regex(/^\d+$/).transform(Number).optional().default('1'),
    limit: z.string().regex(/^\d+$/).transform(Number).optional().default('50'),
    matter_id: z.string().regex(/^\d+$/).transform(Number).optional(),
    lawyer_id: z.string().regex(/^\d+$/).transform(Number).optional(),
    court_location: z.string().optional(),
    start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    status: z.enum(['Scheduled', 'Confirmed', 'Postponed', 'Completed', 'Cancelled']).optional(),
  }),
});

export const getTodaysCalendarSchema = z.object({
  query: z.object({
    lawyer_id: z.string().regex(/^\d+$/).transform(Number).optional(),
  }),
});

export const listCourtsSchema = z.object({
  query: z.object({
    court_level: z.enum(['District Court', 'Circuit Court', 'High Court', 'Court of Appeal', 'Supreme Court']).optional(),
    location: z.string().optional(),
  }),
});

// ====================
// TypeScript Types
// ====================

export type CreateCourtCalendarDto = z.infer<typeof createCourtCalendarSchema>['body'];
export type UpdateCourtCalendarDto = z.infer<typeof updateCourtCalendarSchema>['body'];
export type ListCourtCalendarsQuery = z.infer<typeof listCourtCalendarsSchema>['query'];

export interface CourtCalendarResponse {
  id: number;
  matter_id: number;
  matter_number: string | null;
  matter_title: string;
  client_name: string;
  hearing_id: number | null;
  court_type_id: number | null;
  court_level: string;
  court_location: string;
  court_room: string | null;
  judge_name: string | null;
  docket_number: string | null;
  hearing_date: string;
  hearing_time: string | null;
  hearing_type: string;
  duration_minutes: number | null;
  status: string;
  assigned_lawyer_id: number;
  lawyer_name: string;
  notes: string | null;
  postponement_reason: string | null;
  outcome: string | null;
  created_by: number | null;
  created_at: string;
  updated_at: string;
}

export interface CourtTypeResponse {
  id: number;
  court_level: string;
  location: string;
  court_name: string;
  address: string | null;
  contact_phone: string | null;
  contact_email: string | null;
  is_active: boolean;
}

export interface TodaysCalendarItem {
  id: number;
  matter_id: number;
  matter_number: string | null;
  matter_title: string;
  court_level: string;
  court_location: string;
  court_room: string | null;
  judge_name: string | null;
  hearing_date: string;
  hearing_time: string | null;
  hearing_type: string;
  status: string;
  assigned_lawyer_id: number;
  lawyer_name: string;
  client_name: string;
}
