import { getPool } from '../../db/pool.js';
import {
  CreateCourtCalendarDto,
  UpdateCourtCalendarDto,
  CourtCalendarResponse,
  CourtTypeResponse,
  TodaysCalendarItem,
  ListCourtCalendarsQuery,
} from './dto.js';

export class CourtRepository {
  /**
   * Get all court calendars with pagination and filters
   */
  async findAll(query: ListCourtCalendarsQuery): Promise<{ calendars: CourtCalendarResponse[]; total: number }> {
    const pool = await getPool();
    const { page = 1, limit = 50, matter_id, lawyer_id, court_location, start_date, end_date, status } = query;
    const offset = (page - 1) * limit;

    let whereConditions: string[] = [];
    const params: any[] = [];

    if (matter_id) {
      whereConditions.push('cc.matter_id = ?');
      params.push(matter_id);
    }

    if (lawyer_id) {
      whereConditions.push('cc.assigned_lawyer_id = ?');
      params.push(lawyer_id);
    }

    if (court_location) {
      whereConditions.push('cc.court_location LIKE ?');
      params.push(`%${court_location}%`);
    }

    if (start_date) {
      whereConditions.push('cc.hearing_date >= ?');
      params.push(start_date);
    }

    if (end_date) {
      whereConditions.push('cc.hearing_date <= ?');
      params.push(end_date);
    }

    if (status) {
      whereConditions.push('cc.status = ?');
      params.push(status);
    }

    const whereClause = whereConditions.length > 0 ? `WHERE ${whereConditions.join(' AND ')}` : '';

    // Get total count
    const [countResult] = await pool.query<any[]>(
      `SELECT COUNT(*) as total FROM court_calendars cc ${whereClause}`,
      params
    );
    const total = countResult[0].total;

    // Get paginated results
    const [rows] = await pool.query<any[]>(
      `SELECT 
        cc.id, cc.matter_id, c.matter_id as matter_number, c.title as matter_title,
        cl.name as client_name, cc.hearing_id, cc.court_type_id,
        cc.court_level, cc.court_location, cc.court_room, cc.judge_name,
        cc.docket_number, cc.hearing_date, cc.hearing_time, cc.hearing_type,
        cc.duration_minutes, cc.status, cc.assigned_lawyer_id,
        CONCAT(e.first_name, ' ', e.surname) as lawyer_name,
        cc.notes, cc.postponement_reason, cc.outcome,
        cc.created_by, cc.created_at, cc.updated_at
       FROM court_calendars cc
       JOIN cases c ON c.id = cc.matter_id
       JOIN clients cl ON cl.id = c.client_id
       JOIN lawyers l ON l.id = cc.assigned_lawyer_id
       JOIN employees e ON e.id = l.employee_id
       ${whereClause}
       ORDER BY cc.hearing_date ASC, cc.hearing_time ASC
       LIMIT ? OFFSET ?`,
      [...params, limit, offset]
    );

    return { calendars: rows, total };
  }

  /**
   * Find court calendar by ID
   */
  async findById(id: number): Promise<CourtCalendarResponse | null> {
    const pool = await getPool();

    const [rows] = await pool.query<any[]>(
      `SELECT 
        cc.id, cc.matter_id, c.matter_id as matter_number, c.title as matter_title,
        cl.name as client_name, cc.hearing_id, cc.court_type_id,
        cc.court_level, cc.court_location, cc.court_room, cc.judge_name,
        cc.docket_number, cc.hearing_date, cc.hearing_time, cc.hearing_type,
        cc.duration_minutes, cc.status, cc.assigned_lawyer_id,
        CONCAT(e.first_name, ' ', e.surname) as lawyer_name,
        cc.notes, cc.postponement_reason, cc.outcome,
        cc.created_by, cc.created_at, cc.updated_at
       FROM court_calendars cc
       JOIN cases c ON c.id = cc.matter_id
       JOIN clients cl ON cl.id = c.client_id
       JOIN lawyers l ON l.id = cc.assigned_lawyer_id
       JOIN employees e ON e.id = l.employee_id
       WHERE cc.id = ?`,
      [id]
    );

    return rows.length > 0 ? rows[0] : null;
  }

  /**
   * Create court calendar entry
   */
  async create(data: CreateCourtCalendarDto, userId: number): Promise<number> {
    const pool = await getPool();

    const [result] = await pool.query<any>(
      `INSERT INTO court_calendars (
        matter_id, hearing_id, court_type_id, court_level, court_location,
        court_room, judge_name, docket_number, hearing_date, hearing_time,
        hearing_type, duration_minutes, status, assigned_lawyer_id,
        notes, created_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Scheduled', ?, ?, ?)`,
      [
        data.matter_id,
        data.hearing_id || null,
        data.court_type_id || null,
        data.court_level,
        data.court_location,
        data.court_room || null,
        data.judge_name || null,
        data.docket_number || null,
        data.hearing_date,
        data.hearing_time || null,
        data.hearing_type,
        data.duration_minutes || 60,
        data.assigned_lawyer_id,
        data.notes || null,
        userId,
      ]
    );

    return result.insertId;
  }

  /**
   * Update court calendar entry
   */
  async update(id: number, data: UpdateCourtCalendarDto): Promise<boolean> {
    const pool = await getPool();

    const fields: string[] = [];
    const values: any[] = [];

    if (data.court_type_id !== undefined) {
      fields.push('court_type_id = ?');
      values.push(data.court_type_id);
    }
    if (data.court_level !== undefined) {
      fields.push('court_level = ?');
      values.push(data.court_level);
    }
    if (data.court_location !== undefined) {
      fields.push('court_location = ?');
      values.push(data.court_location);
    }
    if (data.court_room !== undefined) {
      fields.push('court_room = ?');
      values.push(data.court_room);
    }
    if (data.judge_name !== undefined) {
      fields.push('judge_name = ?');
      values.push(data.judge_name);
    }
    if (data.docket_number !== undefined) {
      fields.push('docket_number = ?');
      values.push(data.docket_number);
    }
    if (data.hearing_date !== undefined) {
      fields.push('hearing_date = ?');
      values.push(data.hearing_date);
    }
    if (data.hearing_time !== undefined) {
      fields.push('hearing_time = ?');
      values.push(data.hearing_time);
    }
    if (data.hearing_type !== undefined) {
      fields.push('hearing_type = ?');
      values.push(data.hearing_type);
    }
    if (data.duration_minutes !== undefined) {
      fields.push('duration_minutes = ?');
      values.push(data.duration_minutes);
    }
    if (data.status !== undefined) {
      fields.push('status = ?');
      values.push(data.status);
    }
    if (data.assigned_lawyer_id !== undefined) {
      fields.push('assigned_lawyer_id = ?');
      values.push(data.assigned_lawyer_id);
    }
    if (data.notes !== undefined) {
      fields.push('notes = ?');
      values.push(data.notes);
    }
    if (data.postponement_reason !== undefined) {
      fields.push('postponement_reason = ?');
      values.push(data.postponement_reason);
    }
    if (data.outcome !== undefined) {
      fields.push('outcome = ?');
      values.push(data.outcome);
    }

    if (fields.length === 0) return false;

    values.push(id);

    const [result] = await pool.query<any>(
      `UPDATE court_calendars SET ${fields.join(', ')} WHERE id = ?`,
      values
    );

    return result.affectedRows > 0;
  }

  /**
   * Delete court calendar entry
   */
  async delete(id: number): Promise<boolean> {
    const pool = await getPool();

    const [result] = await pool.query<any>(
      `DELETE FROM court_calendars WHERE id = ?`,
      [id]
    );

    return result.affectedRows > 0;
  }

  /**
   * Get today's calendar (from view)
   */
  async getTodaysCalendar(lawyerId?: number): Promise<TodaysCalendarItem[]> {
    const pool = await getPool();

    let query = `SELECT * FROM vw_todays_calendar`;
    const params: any[] = [];

    if (lawyerId) {
      query += ` WHERE assigned_lawyer_id = ?`;
      params.push(lawyerId);
    }

    query += ` ORDER BY hearing_time ASC`;

    const [rows] = await pool.query<any[]>(query, params);

    return rows;
  }

  /**
   * Check for scheduling clashes (same lawyer, overlapping time)
   */
  async checkClashes(
    lawyerId: number,
    hearingDate: string,
    hearingTime: string,
    durationMinutes: number,
    excludeId?: number
  ): Promise<CourtCalendarResponse[]> {
    const pool = await getPool();

    // Calculate end time
    const [hours, minutes] = hearingTime.split(':').map(Number);
    const startMinutes = hours * 60 + minutes;
    const endMinutes = startMinutes + durationMinutes;
    const endHours = Math.floor(endMinutes / 60);
    const endMins = endMinutes % 60;
    const endTime = `${String(endHours).padStart(2, '0')}:${String(endMins).padStart(2, '0')}:00`;

    const [rows] = await pool.query<any[]>(
      `SELECT 
        cc.id, cc.matter_id, c.matter_id as matter_number, c.title as matter_title,
        cl.name as client_name, cc.court_level, cc.court_location,
        cc.hearing_date, cc.hearing_time, cc.duration_minutes,
        cc.assigned_lawyer_id, CONCAT(e.first_name, ' ', e.surname) as lawyer_name
       FROM court_calendars cc
       JOIN cases c ON c.id = cc.matter_id
       JOIN clients cl ON cl.id = c.client_id
       JOIN lawyers l ON l.id = cc.assigned_lawyer_id
       JOIN employees e ON e.id = l.employee_id
       WHERE cc.assigned_lawyer_id = ?
         AND cc.hearing_date = ?
         AND cc.status IN ('Scheduled', 'Confirmed')
         AND cc.hearing_time IS NOT NULL
         ${excludeId ? 'AND cc.id != ?' : ''}
         AND (
           (cc.hearing_time >= ? AND cc.hearing_time < ?) OR
           (ADDTIME(cc.hearing_time, SEC_TO_TIME(cc.duration_minutes * 60)) > ? AND ADDTIME(cc.hearing_time, SEC_TO_TIME(cc.duration_minutes * 60)) <= ?)
         )`,
      excludeId
        ? [lawyerId, hearingDate, excludeId, hearingTime, endTime, hearingTime, endTime]
        : [lawyerId, hearingDate, hearingTime, endTime, hearingTime, endTime]
    );

    return rows;
  }

  /**
   * Get all court types
   */
  async getCourtTypes(courtLevel?: string, location?: string): Promise<CourtTypeResponse[]> {
    const pool = await getPool();

    let whereConditions: string[] = ['is_active = 1'];
    const params: any[] = [];

    if (courtLevel) {
      whereConditions.push('court_level = ?');
      params.push(courtLevel);
    }

    if (location) {
      whereConditions.push('location LIKE ?');
      params.push(`%${location}%`);
    }

    const whereClause = `WHERE ${whereConditions.join(' AND ')}`;

    const [rows] = await pool.query<any[]>(
      `SELECT id, court_level, location, court_name, address, contact_phone, contact_email, is_active
       FROM court_types
       ${whereClause}
       ORDER BY court_level, location, court_name`,
      params
    );

    return rows;
  }
}
