import { Router, Request, Response, NextFunction } from 'express';
import { CourtRepository } from './repo.js';
import {
  createCourtCalendarSchema,
  updateCourtCalendarSchema,
  getCourtCalendarSchema,
  listCourtCalendarsSchema,
  getTodaysCalendarSchema,
  listCourtsSchema,
} from './dto.js';
import { validate } from '../../middleware/validate.js';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { logger } from '../../utils/logger.js';

const router = Router();
const repo = new CourtRepository();

// All routes require authentication
router.use(authenticate);

/**
 * GET /api/courts/calendars
 * List court calendars with pagination and filters
 */
router.get(
  '/calendars',
  requirePermission('courts:read'),
  validate(listCourtCalendarsSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const query = req.query as any;
      const { calendars, total } = await repo.findAll(query);

      const page = parseInt(query.page || '1');
      const limit = parseInt(query.limit || '50');
      const totalPages = Math.ceil(total / limit);

      res.json({
        success: true,
        data: {
          calendars,
          pagination: {
            page,
            limit,
            total,
            totalPages,
          },
        },
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id }, 'Failed to list court calendars');
      next(error);
    }
  }
);

/**
 * GET /api/courts/calendars/today
 * Get today's court calendar
 */
router.get(
  '/calendars/today',
  requirePermission('courts:read'),
  validate(getTodaysCalendarSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { lawyer_id } = req.query;
      const lawyerId = lawyer_id ? parseInt(lawyer_id as string) : undefined;

      const calendar = await repo.getTodaysCalendar(lawyerId);

      res.json({
        success: true,
        data: { calendar },
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id }, 'Failed to get todays calendar');
      next(error);
    }
  }
);

/**
 * GET /api/courts/calendars/:id
 * Get court calendar by ID
 */
router.get(
  '/calendars/:id',
  requirePermission('courts:read'),
  validate(getCourtCalendarSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { id } = req.params;
      const calendar = await repo.findById(parseInt(id));

      if (!calendar) {
        return res.status(404).json({
          success: false,
          error: 'Court calendar not found',
        });
      }

      res.json({
        success: true,
        data: calendar,
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id, calendarId: req.params.id }, 'Failed to get court calendar');
      next(error);
    }
  }
);

/**
 * POST /api/courts/calendars
 * Create court calendar entry
 */
router.post(
  '/calendars',
  requirePermission('courts:create'),
  validate(createCourtCalendarSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const data = req.body;
      const userId = req.user!.id;

      // Check for scheduling clashes if time is provided
      if (data.hearing_time) {
        const clashes = await repo.checkClashes(
          data.assigned_lawyer_id,
          data.hearing_date,
          data.hearing_time,
          data.duration_minutes || 60
        );

        if (clashes.length > 0) {
          return res.status(409).json({
            success: false,
            error: 'Scheduling clash detected',
            data: {
              message: 'The lawyer has another hearing at the same time',
              clashes,
            },
          });
        }
      }

      const calendarId = await repo.create(data, userId);

      // Log audit
      logger.info({
        userId,
        action: 'CREATE_COURT_CALENDAR',
        calendarId,
        matterId: data.matter_id,
      }, 'Court calendar created');

      const calendar = await repo.findById(calendarId);

      res.status(201).json({
        success: true,
        message: 'Court calendar created successfully',
        data: calendar,
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id }, 'Failed to create court calendar');
      next(error);
    }
  }
);

/**
 * PUT /api/courts/calendars/:id
 * Update court calendar entry
 */
router.put(
  '/calendars/:id',
  requirePermission('courts:update'),
  validate(updateCourtCalendarSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { id } = req.params;
      const data = req.body;

      // Check for scheduling clashes if time/date is being updated
      if (data.hearing_time && data.hearing_date && data.assigned_lawyer_id) {
        const clashes = await repo.checkClashes(
          data.assigned_lawyer_id,
          data.hearing_date,
          data.hearing_time,
          data.duration_minutes || 60,
          parseInt(id)
        );

        if (clashes.length > 0) {
          return res.status(409).json({
            success: false,
            error: 'Scheduling clash detected',
            data: {
              message: 'The lawyer has another hearing at the same time',
              clashes,
            },
          });
        }
      }

      const updated = await repo.update(parseInt(id), data);

      if (!updated) {
        return res.status(404).json({
          success: false,
          error: 'Court calendar not found or no changes made',
        });
      }

      // Log audit
      logger.info({
        userId: req.user!.id,
        action: 'UPDATE_COURT_CALENDAR',
        calendarId: id,
        changes: data,
      }, 'Court calendar updated');

      const calendar = await repo.findById(parseInt(id));

      res.json({
        success: true,
        message: 'Court calendar updated successfully',
        data: calendar,
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id, calendarId: req.params.id }, 'Failed to update court calendar');
      next(error);
    }
  }
);

/**
 * DELETE /api/courts/calendars/:id
 * Delete court calendar entry
 */
router.delete(
  '/calendars/:id',
  requirePermission('courts:delete'),
  validate(getCourtCalendarSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { id } = req.params;

      const deleted = await repo.delete(parseInt(id));

      if (!deleted) {
        return res.status(404).json({
          success: false,
          error: 'Court calendar not found',
        });
      }

      // Log audit
      logger.info({
        userId: req.user!.id,
        action: 'DELETE_COURT_CALENDAR',
        calendarId: id,
      }, 'Court calendar deleted');

      res.json({
        success: true,
        message: 'Court calendar deleted successfully',
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id, calendarId: req.params.id }, 'Failed to delete court calendar');
      next(error);
    }
  }
);

/**
 * GET /api/courts/types
 * Get list of Ghana court types
 */
router.get(
  '/types',
  requirePermission('courts:read'),
  validate(listCourtsSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { court_level, location } = req.query;

      const courts = await repo.getCourtTypes(
        court_level as string | undefined,
        location as string | undefined
      );

      res.json({
        success: true,
        data: { courts },
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id }, 'Failed to get court types');
      next(error);
    }
  }
);

export default router;
