import { getPool } from '../../db/pool.js';
import { CreateDepartmentDto, UpdateDepartmentDto, DepartmentResponse } from './dto.js';

export class DepartmentRepository {
  /**
   * Get all departments
   */
  async findAll(): Promise<DepartmentResponse[]> {
    const pool = await getPool();
    
    const [rows] = await pool.query<any[]>(
      `SELECT id, name, description, created_at
       FROM departments
       ORDER BY name ASC`
    );

    return rows;
  }

  /**
   * Get department by ID
   */
  async findById(id: number): Promise<DepartmentResponse | null> {
    const pool = await getPool();
    
    const [rows] = await pool.query<any[]>(
      `SELECT id, name, description, created_at
       FROM departments
       WHERE id = ?`,
      [id]
    );

    return rows.length > 0 ? rows[0] : null;
  }

  /**
   * Check if department name exists (excluding specific ID for updates)
   */
  async nameExists(name: string, excludeId?: number): Promise<boolean> {
    const pool = await getPool();
    
    const query = excludeId
      ? 'SELECT COUNT(*) as count FROM departments WHERE name = ? AND id != ?'
      : 'SELECT COUNT(*) as count FROM departments WHERE name = ?';
    
    const params = excludeId ? [name, excludeId] : [name];
    
    const [rows] = await pool.query<any[]>(query, params);
    
    return rows[0].count > 0;
  }

  /**
   * Create department
   */
  async create(data: CreateDepartmentDto): Promise<DepartmentResponse> {
    const pool = await getPool();
    
    const [result] = await pool.query<any>(
      `INSERT INTO departments (name, description)
       VALUES (?, ?)`,
      [data.name, data.description || null]
    );

    const newDept = await this.findById(result.insertId);
    if (!newDept) {
      throw new Error('Failed to create department');
    }

    return newDept;
  }

  /**
   * Update department
   */
  async update(id: number, data: UpdateDepartmentDto): Promise<DepartmentResponse> {
    const pool = await getPool();
    
    const updates: string[] = [];
    const values: any[] = [];

    if (data.name !== undefined) {
      updates.push('name = ?');
      values.push(data.name);
    }

    if (data.description !== undefined) {
      updates.push('description = ?');
      values.push(data.description);
    }

    if (updates.length === 0) {
      const dept = await this.findById(id);
      if (!dept) {
        throw new Error('Department not found');
      }
      return dept;
    }

    values.push(id);

    await pool.query(
      `UPDATE departments SET ${updates.join(', ')} WHERE id = ?`,
      values
    );

    const updated = await this.findById(id);
    if (!updated) {
      throw new Error('Department not found');
    }

    return updated;
  }

  /**
   * Delete department
   */
  async delete(id: number): Promise<void> {
    const pool = await getPool();
    
    await pool.query('DELETE FROM departments WHERE id = ?', [id]);
  }

  /**
   * Check if department has associated roles
   */
  async hasRoles(id: number): Promise<boolean> {
    const pool = await getPool();
    
    const [rows] = await pool.query<any[]>(
      'SELECT COUNT(*) as count FROM roles WHERE department_id = ?',
      [id]
    );

    return rows[0].count > 0;
  }

  /**
   * Log department action to audit log
   */
  async logAction(
    userId: number,
    departmentId: number,
    action: 'CREATE' | 'UPDATE' | 'DELETE',
    meta: Record<string, any>
  ): Promise<void> {
    const pool = await getPool();
    
    await pool.query(
      `INSERT INTO audit_logs (user_id, entity, entity_id, action, meta)
       VALUES (?, 'department', ?, ?, ?)`,
      [userId, departmentId, action, JSON.stringify(meta)]
    );
  }
}
