import { z } from 'zod';

// Create employee DTO
export const createEmployeeSchema = z.object({
  body: z.object({
    employee_number: z.string().min(1, 'Employee number is required').max(50),
    first_name: z.string().min(1, 'First name is required').max(100),
    middle_name: z.string().max(100).optional(),
    surname: z.string().min(1, 'Surname is required').max(100),
    dob: z.string().regex(/^\d{4}-\d{2}-\d{2}$/, 'Invalid date format (YYYY-MM-DD)'),
    phone: z.string().min(1, 'Phone is required').max(40),
    email: z.string().email('Invalid email address').max(190),
    address: z.string().min(1, 'Address is required'),
    department_id: z.number().int().positive('Department is required'),
    role_id: z.number().int().positive('Role is required'),
    is_active: z.boolean().optional(),
  }),
});

// Update employee DTO
export const updateEmployeeSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/, 'Invalid employee ID'),
  }),
  body: z.object({
    employee_number: z.string().min(1).max(50).optional(),
    first_name: z.string().min(1).max(100).optional(),
    middle_name: z.string().max(100).optional(),
    surname: z.string().min(1).max(100).optional(),
    dob: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    phone: z.string().min(1).max(40).optional(),
    email: z.string().email().max(190).optional(),
    address: z.string().min(1).optional(),
    department_id: z.number().int().positive().optional(),
    role_id: z.number().int().positive().optional(),
    is_active: z.boolean().optional(),
  }),
});

// Delete employee DTO
export const deleteEmployeeSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/, 'Invalid employee ID'),
  }),
});

// Get employee DTO
export const getEmployeeSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/, 'Invalid employee ID'),
  }),
});

export type CreateEmployeeDto = z.infer<typeof createEmployeeSchema>['body'];
export type UpdateEmployeeDto = z.infer<typeof updateEmployeeSchema>['body'];

export interface EmployeeResponse {
  id: number;
  employee_number: string;
  first_name: string;
  middle_name: string | null;
  surname: string;
  dob: string;
  phone: string;
  email: string;
  address: string;
  department_id: number;
  department_name: string;
  role_id: number;
  role_name: string;
  is_active: boolean;
  created_at: Date;
}
