import { Router, Request, Response, NextFunction } from 'express';
import { EmployeeRepository } from './repo.js';
import {
  createEmployeeSchema,
  updateEmployeeSchema,
  deleteEmployeeSchema,
  getEmployeeSchema,
} from './dto.js';
import { validate } from '../../middleware/validate.js';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { AppError } from '../../middleware/errors.js';

const router = Router();
const employeeRepo = new EmployeeRepository();

// All routes require authentication
router.use(authenticate);

/**
 * GET /api/employees
 * List all employees
 */
router.get(
  '/',
  requirePermission('employees:read'),
  async (req: Request, res: Response, next: NextFunction): Promise<void> => {
    try {
      const employees = await employeeRepo.findAll();

      res.json({
        success: true,
        data: { employees },
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/employees/:id
 * Get employee by ID
 */
router.get(
  '/:id',
  requirePermission('employees:read'),
  validate(getEmployeeSchema),
  async (req: Request, res: Response, next: NextFunction): Promise<void> => {
    try {
      const id = parseInt(req.params.id);
      const employee = await employeeRepo.findById(id);

      if (!employee) {
        throw new AppError(404, 'Employee not found');
      }

      res.json({
        success: true,
        data: { employee },
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/employees
 * Create new employee (Admin/Manager only)
 */
router.post(
  '/',
  requirePermission('employees:create'),
  validate(createEmployeeSchema),
  async (req: Request, res: Response, next: NextFunction): Promise<void> => {
    try {
      const {
        employee_number,
        first_name,
        middle_name,
        surname,
        dob,
        phone,
        email,
        address,
        department_id,
        role_id,
        is_active,
      } = req.body;

      // Check if email exists
      const emailExists = await employeeRepo.emailExists(email);
      if (emailExists) {
        throw new AppError(409, 'Email already exists');
      }

      // Check if department exists
      const deptExists = await employeeRepo.departmentExists(department_id);
      if (!deptExists) {
        throw new AppError(404, 'Department not found');
      }

      // Check if role exists and belongs to department
      const roleValid = await employeeRepo.roleExistsInDepartment(role_id, department_id);
      if (!roleValid) {
        throw new AppError(400, 'Role does not belong to the selected department');
      }

      const employee = await employeeRepo.create({
        employee_number,
        first_name,
        middle_name,
        surname,
        dob,
        phone,
        email,
        address,
        department_id,
        role_id,
        is_active,
      });

      // Log action
      await employeeRepo.logAction(req.user!.id, employee.id, 'CREATE', {
        name: `${employee.first_name} ${employee.surname}`,
        email: employee.email,
        department_id: employee.department_id,
        role_id: employee.role_id,
      });

      res.status(201).json({
        success: true,
        data: { employee },
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * PUT /api/employees/:id
 * Update employee (Admin/Manager only)
 */
router.put(
  '/:id',
  requirePermission('employees:update'),
  validate(updateEmployeeSchema),
  async (req: Request, res: Response, next: NextFunction): Promise<void> => {
    try {
      const id = parseInt(req.params.id);
      const {
        employee_number,
        first_name,
        middle_name,
        surname,
        dob,
        phone,
        email,
        address,
        department_id,
        role_id,
        is_active,
      } = req.body;

      // Check if employee exists
      const existing = await employeeRepo.findById(id);
      if (!existing) {
        throw new AppError(404, 'Employee not found');
      }

      // Check for duplicate email (if email is being changed)
      if (email && email !== existing.email) {
        const emailExists = await employeeRepo.emailExists(email, id);
        if (emailExists) {
          throw new AppError(409, 'Email already exists');
        }
      }

      // If department is changing, check if it exists
      if (department_id && department_id !== existing.department_id) {
        const deptExists = await employeeRepo.departmentExists(department_id);
        if (!deptExists) {
          throw new AppError(404, 'Department not found');
        }
      }

      // Check if role is valid for department
      const targetDeptId = department_id || existing.department_id;
      const targetRoleId = role_id || existing.role_id;
      
      if (role_id || department_id) {
        const roleValid = await employeeRepo.roleExistsInDepartment(
          targetRoleId,
          targetDeptId
        );
        if (!roleValid) {
          throw new AppError(400, 'Role does not belong to the selected department');
        }
      }

      const employee = await employeeRepo.update(id, {
        employee_number,
        first_name,
        middle_name,
        surname,
        dob,
        phone,
        email,
        address,
        department_id,
        role_id,
        is_active,
      });

      // Log action
      await employeeRepo.logAction(req.user!.id, id, 'UPDATE', {
        old: existing,
        new: employee,
      });

      res.json({
        success: true,
        data: { employee },
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * DELETE /api/employees/:id
 * Delete employee (Admin only)
 */
router.delete(
  '/:id',
  requirePermission('employees:delete'),
  validate(deleteEmployeeSchema),
  async (req: Request, res: Response, next: NextFunction): Promise<void> => {
    try {
      const id = parseInt(req.params.id);

      // Check if employee exists
      const employee = await employeeRepo.findById(id);
      if (!employee) {
        throw new AppError(404, 'Employee not found');
      }

      // Check if employee has associated user
      const hasUser = await employeeRepo.hasUser(id);
      if (hasUser) {
        throw new AppError(
          400,
          'Cannot delete employee with associated user account. Delete user first.'
        );
      }

      await employeeRepo.delete(id);

      // Log action
      await employeeRepo.logAction(req.user!.id, id, 'DELETE', {
        name: `${employee.first_name} ${employee.surname}`,
        email: employee.email,
      });

      res.json({
        success: true,
        data: { message: 'Employee deleted successfully' },
      });
    } catch (error) {
      next(error);
    }
  }
);

export default router;
