import { z } from 'zod';

// ====================
// Request Schemas
// ====================

export const createEvidenceSchema = z.object({
  body: z.object({
    matter_id: z.number().int().positive(),
    exhibit_label: z.string().min(1).max(50),
    description: z.string().min(5),
    evidence_type: z.enum(['Document', 'Physical Object', 'Digital File', 'Audio Recording', 'Video Recording', 'Photograph', 'Other']),
    file_path: z.string().max(500).optional(),
    file_hash: z.string().max(128).optional(),
    file_size_bytes: z.number().int().positive().optional(),
    file_mime_type: z.string().max(100).optional(),
    collection_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    collection_location: z.string().max(300).optional(),
    collected_by: z.string().max(200).optional(),
    storage_location: z.string().max(300).optional(),
    is_sealed: z.boolean().default(false),
    seal_number: z.string().max(100).optional(),
    linked_hearing_id: z.number().int().positive().optional(),
    notes: z.string().optional(),
  }),
});

export const updateEvidenceSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
  body: z.object({
    exhibit_label: z.string().min(1).max(50).optional(),
    description: z.string().min(5).optional(),
    evidence_type: z.enum(['Document', 'Physical Object', 'Digital File', 'Audio Recording', 'Video Recording', 'Photograph', 'Other']).optional(),
    file_path: z.string().max(500).optional(),
    file_hash: z.string().max(128).optional(),
    file_size_bytes: z.number().int().positive().optional(),
    file_mime_type: z.string().max(100).optional(),
    collection_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    collection_location: z.string().max(300).optional(),
    collected_by: z.string().max(200).optional(),
    current_state: z.enum(['Collected', 'Labeled', 'Stored', 'In Transit', 'Presented', 'Returned', 'Disposed']).optional(),
    storage_location: z.string().max(300).optional(),
    is_sealed: z.boolean().optional(),
    seal_number: z.string().max(100).optional(),
    linked_hearing_id: z.number().int().positive().optional(),
    notes: z.string().optional(),
  }),
});

export const getEvidenceSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
});

export const listEvidenceSchema = z.object({
  query: z.object({
    matter_id: z.string().regex(/^\d+$/).transform(Number).optional(),
    evidence_type: z.enum(['Document', 'Physical Object', 'Digital File', 'Audio Recording', 'Video Recording', 'Photograph', 'Other']).optional(),
    current_state: z.enum(['Collected', 'Labeled', 'Stored', 'In Transit', 'Presented', 'Returned', 'Disposed']).optional(),
    search: z.string().optional(),
  }),
});

export const recordCustodyEventSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
  body: z.object({
    event_type: z.enum(['Collected', 'Labeled', 'Transferred', 'Stored', 'Retrieved', 'Presented', 'Returned', 'Disposed', 'Photographed', 'Examined']),
    from_person: z.string().max(200).optional(),
    to_person: z.string().max(200).optional(),
    location: z.string().max(300).optional(),
    purpose: z.string().optional(),
    witness_name: z.string().max(200).optional(),
    notes: z.string().optional(),
  }),
});

export const getCustodyHistorySchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
});

// ====================
// TypeScript Types
// ====================

export type CreateEvidenceDto = z.infer<typeof createEvidenceSchema>['body'];
export type UpdateEvidenceDto = z.infer<typeof updateEvidenceSchema>['body'];
export type RecordCustodyEventDto = z.infer<typeof recordCustodyEventSchema>['body'];

export interface EvidenceResponse {
  id: number;
  matter_id: number;
  matter_number: string | null;
  matter_title: string;
  exhibit_label: string;
  description: string;
  evidence_type: string;
  file_path: string | null;
  file_hash: string | null;
  file_size_bytes: number | null;
  file_mime_type: string | null;
  collection_date: string | null;
  collection_location: string | null;
  collected_by: string | null;
  current_state: string;
  storage_location: string | null;
  is_sealed: boolean;
  seal_number: string | null;
  linked_hearing_id: number | null;
  notes: string | null;
  created_by: number | null;
  created_at: string;
  updated_at: string;
}

export interface CustodyEventResponse {
  id: number;
  evidence_id: number;
  event_type: string;
  event_date: string;
  from_person: string | null;
  to_person: string | null;
  location: string | null;
  purpose: string | null;
  witness_name: string | null;
  notes: string | null;
  recorded_by: number;
  recorded_by_name: string;
  recorded_at: string;
}

export interface EvidenceWithCustody extends EvidenceResponse {
  custody_events: CustodyEventResponse[];
}
