import { z } from 'zod';

// ================================================
// Expense Entry DTOs
// ================================================

// Create Expense Entry
export const createExpenseSchema = z.object({
  matter_id: z.number().int().positive(),
  expense_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  category: z.string().min(1).max(100),
  amount: z.number().positive(),
  currency: z.string().length(3).default('GHS'),
  description: z.string().min(1).max(5000),
  receipt_path: z.string().max(500).optional(),
  receipt_number: z.string().max(100).optional(),
  vendor_name: z.string().max(200).optional(),
  billable: z.boolean().default(true),
  markup_percent: z.number().min(0).max(100).default(0),
  reimbursable: z.boolean().default(false),
});

export type CreateExpenseDto = z.infer<typeof createExpenseSchema>;

// Update Expense Entry
export const updateExpenseSchema = z.object({
  expense_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  category: z.string().min(1).max(100).optional(),
  amount: z.number().positive().optional(),
  description: z.string().min(1).max(5000).optional(),
  receipt_path: z.string().max(500).optional(),
  receipt_number: z.string().max(100).optional(),
  vendor_name: z.string().max(200).optional(),
  billable: z.boolean().optional(),
  markup_percent: z.number().min(0).max(100).optional(),
  reimbursable: z.boolean().optional(),
});

export type UpdateExpenseDto = z.infer<typeof updateExpenseSchema>;

// Query Parameters
export const queryExpensesSchema = z.object({
  page: z.string().optional().transform(val => val ? parseInt(val) : 1),
  limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
  user_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  matter_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  category: z.string().optional(),
  status: z.enum(['Draft', 'Submitted', 'Approved', 'Rejected', 'Invoiced', 'Reimbursed']).optional(),
  start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  billable: z.string().optional().transform(val => val === 'true' ? true : val === 'false' ? false : undefined),
  reimbursable: z.string().optional().transform(val => val === 'true' ? true : val === 'false' ? false : undefined),
});

export type QueryExpensesDto = z.infer<typeof queryExpensesSchema>;

// Submit for Approval
export const submitExpenseSchema = z.object({
  expense_ids: z.array(z.number().int().positive()).min(1),
});

export type SubmitExpenseDto = z.infer<typeof submitExpenseSchema>;

// Approve Expense
export const approveExpenseSchema = z.object({
  expense_ids: z.array(z.number().int().positive()).min(1),
});

export type ApproveExpenseDto = z.infer<typeof approveExpenseSchema>;

// Reject Expense
export const rejectExpenseSchema = z.object({
  expense_ids: z.array(z.number().int().positive()).min(1),
  rejection_reason: z.string().min(1).max(1000),
});

export type RejectExpenseDto = z.infer<typeof rejectExpenseSchema>;

// ================================================
// Expense Category DTOs
// ================================================

// Create Expense Category (admin only)
export const createExpenseCategorySchema = z.object({
  category_name: z.string().min(1).max(100),
  description: z.string().max(500).optional(),
  default_billable: z.boolean().default(true),
  default_markup_percent: z.number().min(0).max(100).default(0),
  sort_order: z.number().int().default(0),
});

export type CreateExpenseCategoryDto = z.infer<typeof createExpenseCategorySchema>;

// Update Expense Category
export const updateExpenseCategorySchema = z.object({
  category_name: z.string().min(1).max(100).optional(),
  description: z.string().max(500).optional(),
  default_billable: z.boolean().optional(),
  default_markup_percent: z.number().min(0).max(100).optional(),
  is_active: z.boolean().optional(),
  sort_order: z.number().int().optional(),
});

export type UpdateExpenseCategoryDto = z.infer<typeof updateExpenseCategorySchema>;

// ================================================
// Response Types
// ================================================

export interface ExpenseResponse {
  id: number;
  matter_id: number;
  matter_number: string | null;
  matter_title: string;
  user_id: number;
  user_name: string;
  expense_date: string;
  category: string;
  amount: number;
  currency: string;
  description: string;
  receipt_path: string | null;
  receipt_number: string | null;
  vendor_name: string | null;
  billable: boolean;
  markup_percent: number;
  billable_amount: number | null;
  reimbursable: boolean;
  status: 'Draft' | 'Submitted' | 'Approved' | 'Rejected' | 'Invoiced' | 'Reimbursed';
  submitted_at: string | null;
  approved_by: number | null;
  approved_by_name: string | null;
  approved_at: string | null;
  rejection_reason: string | null;
  created_at: string;
  updated_at: string;
}

export interface ExpenseCategoryResponse {
  id: number;
  category_name: string;
  description: string | null;
  default_billable: boolean;
  default_markup_percent: number;
  is_active: boolean;
  sort_order: number;
  created_at: string;
  updated_at: string;
}
