import { RowDataPacket, ResultSetHeader } from 'mysql2/promise';
import { getPool } from '../../db/pool.js';
import type {
  CreateExpenseDto,
  UpdateExpenseDto,
  QueryExpensesDto,
  CreateExpenseCategoryDto,
  UpdateExpenseCategoryDto,
  ExpenseResponse,
  ExpenseCategoryResponse,
} from './dto.js';

export class ExpenseRepository {
  // ================================================
  // Expense Entries
  // ================================================

  async listExpenses(query: QueryExpensesDto, currentUserId: number, canViewAll: boolean) {
    const pool = await getPool();
    const { page = 1, limit = 50, user_id, matter_id, category, status, start_date, end_date, billable, reimbursable } = query;
    const offset = (page - 1) * limit;

    let whereConditions: string[] = [];
    let params: any[] = [];

    // Permission check: users can only see their own expenses unless they have view-all permission
    if (!canViewAll) {
      whereConditions.push('ee.user_id = ?');
      params.push(currentUserId);
    } else if (user_id) {
      whereConditions.push('ee.user_id = ?');
      params.push(user_id);
    }

    if (matter_id) {
      whereConditions.push('ee.matter_id = ?');
      params.push(matter_id);
    }

    if (category) {
      whereConditions.push('ee.category = ?');
      params.push(category);
    }

    if (status) {
      whereConditions.push('ee.status = ?');
      params.push(status);
    }

    if (start_date) {
      whereConditions.push('ee.expense_date >= ?');
      params.push(start_date);
    }

    if (end_date) {
      whereConditions.push('ee.expense_date <= ?');
      params.push(end_date);
    }

    if (billable !== undefined) {
      whereConditions.push('ee.billable = ?');
      params.push(billable);
    }

    if (reimbursable !== undefined) {
      whereConditions.push('ee.reimbursable = ?');
      params.push(reimbursable);
    }

    const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';

    // Get total count
    const countQuery = `
      SELECT COUNT(*) as total
      FROM expense_entries ee
      ${whereClause}
    `;
    const [countRows] = await pool.query<RowDataPacket[]>(countQuery, params);
    const total = countRows[0].total;

    // Get expenses
    const query_sql = `
      SELECT 
        ee.*,
        u.username as user_name,
        m.matter_number,
        m.title as matter_title,
        approver.username as approved_by_name
      FROM expense_entries ee
      JOIN users u ON ee.user_id = u.id
      JOIN matters m ON ee.matter_id = m.id
      LEFT JOIN users approver ON ee.approved_by = approver.id
      ${whereClause}
      ORDER BY ee.expense_date DESC, ee.created_at DESC
      LIMIT ? OFFSET ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query_sql, [...params, limit, offset]);

    return {
      expenses: rows as ExpenseResponse[],
      pagination: {
        page,
        limit,
        total,
        totalPages: Math.ceil(total / limit),
      },
    };
  }

  async getExpenseById(id: number, currentUserId: number, canViewAll: boolean) {
    const pool = await getPool();

    const query = `
      SELECT 
        ee.*,
        u.username as user_name,
        m.matter_number,
        m.title as matter_title,
        approver.username as approved_by_name
      FROM expense_entries ee
      JOIN users u ON ee.user_id = u.id
      JOIN matters m ON ee.matter_id = m.id
      LEFT JOIN users approver ON ee.approved_by = approver.id
      WHERE ee.id = ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, [id]);

    if (rows.length === 0) {
      return null;
    }

    const expense = rows[0] as ExpenseResponse;

    // Permission check
    if (!canViewAll && expense.user_id !== currentUserId) {
      throw new Error('Forbidden: Cannot view other users expenses');
    }

    return expense;
  }

  async createExpense(data: CreateExpenseDto, userId: number) {
    const pool = await getPool();

    // Calculate billable_amount if billable
    let billable_amount = null;
    if (data.billable) {
      billable_amount = data.amount * (1 + data.markup_percent / 100);
    }

    const query = `
      INSERT INTO expense_entries (
        matter_id, user_id, expense_date, category, amount, currency,
        description, receipt_path, receipt_number, vendor_name,
        billable, markup_percent, billable_amount, reimbursable,
        created_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;

    const [result] = await pool.query<ResultSetHeader>(query, [
      data.matter_id,
      userId,
      data.expense_date,
      data.category,
      data.amount,
      data.currency,
      data.description,
      data.receipt_path || null,
      data.receipt_number || null,
      data.vendor_name || null,
      data.billable,
      data.markup_percent,
      billable_amount,
      data.reimbursable,
      userId,
    ]);

    return this.getExpenseById(result.insertId, userId, true);
  }

  async updateExpense(id: number, data: UpdateExpenseDto, userId: number, canViewAll: boolean) {
    const pool = await getPool();

    // Check if expense exists and user has permission
    const existing = await this.getExpenseById(id, userId, canViewAll);
    if (!existing) {
      throw new Error('Expense not found');
    }

    // Only allow updates if status is Draft or Rejected
    if (existing.status !== 'Draft' && existing.status !== 'Rejected') {
      throw new Error('Cannot update expense that has been submitted or approved');
    }

    // Only allow user to update their own expenses
    if (existing.user_id !== userId) {
      throw new Error('Forbidden: Cannot update other users expenses');
    }

    const updates: string[] = [];
    const params: any[] = [];

    if (data.expense_date !== undefined) {
      updates.push('expense_date = ?');
      params.push(data.expense_date);
    }
    if (data.category !== undefined) {
      updates.push('category = ?');
      params.push(data.category);
    }
    if (data.amount !== undefined) {
      updates.push('amount = ?');
      params.push(data.amount);
    }
    if (data.description !== undefined) {
      updates.push('description = ?');
      params.push(data.description);
    }
    if (data.receipt_path !== undefined) {
      updates.push('receipt_path = ?');
      params.push(data.receipt_path);
    }
    if (data.receipt_number !== undefined) {
      updates.push('receipt_number = ?');
      params.push(data.receipt_number);
    }
    if (data.vendor_name !== undefined) {
      updates.push('vendor_name = ?');
      params.push(data.vendor_name);
    }
    if (data.billable !== undefined) {
      updates.push('billable = ?');
      params.push(data.billable);
    }
    if (data.markup_percent !== undefined) {
      updates.push('markup_percent = ?');
      params.push(data.markup_percent);
    }
    if (data.reimbursable !== undefined) {
      updates.push('reimbursable = ?');
      params.push(data.reimbursable);
    }

    // Recalculate billable_amount if amount, markup, or billable changed
    if (data.amount !== undefined || data.markup_percent !== undefined || data.billable !== undefined) {
      const amount = data.amount !== undefined ? data.amount : existing.amount;
      const markup = data.markup_percent !== undefined ? data.markup_percent : existing.markup_percent;
      const billable = data.billable !== undefined ? data.billable : existing.billable;
      
      const billable_amount = billable ? amount * (1 + markup / 100) : null;
      updates.push('billable_amount = ?');
      params.push(billable_amount);
    }

    if (updates.length === 0) {
      return existing;
    }

    const query = `UPDATE expense_entries SET ${updates.join(', ')} WHERE id = ?`;
    await pool.query(query, [...params, id]);

    return this.getExpenseById(id, userId, true);
  }

  async deleteExpense(id: number, userId: number, canViewAll: boolean) {
    const pool = await getPool();

    const existing = await this.getExpenseById(id, userId, canViewAll);
    if (!existing) {
      throw new Error('Expense not found');
    }

    // Only allow deletion if status is Draft
    if (existing.status !== 'Draft') {
      throw new Error('Cannot delete expense that has been submitted');
    }

    // Only allow user to delete their own expenses
    if (existing.user_id !== userId) {
      throw new Error('Forbidden: Cannot delete other users expenses');
    }

    await pool.query('DELETE FROM expense_entries WHERE id = ?', [id]);
  }

  async submitExpenses(expenseIds: number[], userId: number) {
    const pool = await getPool();

    // Verify all expenses belong to user and are in Draft status
    const checkQuery = `
      SELECT id, status, user_id 
      FROM expense_entries 
      WHERE id IN (${expenseIds.map(() => '?').join(',')})
    `;
    const [expenses] = await pool.query<RowDataPacket[]>(checkQuery, expenseIds);

    for (const expense of expenses) {
      if (expense.user_id !== userId) {
        throw new Error(`Expense ${expense.id} does not belong to you`);
      }
      if (expense.status !== 'Draft') {
        throw new Error(`Expense ${expense.id} has already been submitted`);
      }
    }

    const updateQuery = `
      UPDATE expense_entries 
      SET status = 'Submitted', submitted_at = NOW()
      WHERE id IN (${expenseIds.map(() => '?').join(',')})
    `;
    await pool.query(updateQuery, expenseIds);
  }

  async approveExpenses(expenseIds: number[], approverId: number) {
    const pool = await getPool();

    // Verify all expenses are in Submitted status
    const checkQuery = `
      SELECT id, status 
      FROM expense_entries 
      WHERE id IN (${expenseIds.map(() => '?').join(',')})
    `;
    const [expenses] = await pool.query<RowDataPacket[]>(checkQuery, expenseIds);

    for (const expense of expenses) {
      if (expense.status !== 'Submitted') {
        throw new Error(`Expense ${expense.id} is not in Submitted status`);
      }
    }

    const updateQuery = `
      UPDATE expense_entries 
      SET status = 'Approved', approved_by = ?, approved_at = NOW()
      WHERE id IN (${expenseIds.map(() => '?').join(',')})
    `;
    await pool.query(updateQuery, [approverId, ...expenseIds]);
  }

  async rejectExpenses(expenseIds: number[], approverId: number, reason: string) {
    const pool = await getPool();

    // Verify all expenses are in Submitted status
    const checkQuery = `
      SELECT id, status 
      FROM expense_entries 
      WHERE id IN (${expenseIds.map(() => '?').join(',')})
    `;
    const [expenses] = await pool.query<RowDataPacket[]>(checkQuery, expenseIds);

    for (const expense of expenses) {
      if (expense.status !== 'Submitted') {
        throw new Error(`Expense ${expense.id} is not in Submitted status`);
      }
    }

    const updateQuery = `
      UPDATE expense_entries 
      SET status = 'Rejected', approved_by = ?, approved_at = NOW(), rejection_reason = ?
      WHERE id IN (${expenseIds.map(() => '?').join(',')})
    `;
    await pool.query(updateQuery, [approverId, reason, ...expenseIds]);
  }

  // ================================================
  // Expense Categories
  // ================================================

  async listExpenseCategories(activeOnly: boolean = true) {
    const pool = await getPool();

    const whereClause = activeOnly ? 'WHERE is_active = TRUE' : '';
    const query = `
      SELECT * FROM expense_categories 
      ${whereClause}
      ORDER BY sort_order, category_name
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query);
    return rows as ExpenseCategoryResponse[];
  }

  async getExpenseCategoryById(id: number) {
    const pool = await getPool();

    const [rows] = await pool.query<RowDataPacket[]>(
      'SELECT * FROM expense_categories WHERE id = ?',
      [id]
    );

    return rows.length > 0 ? (rows[0] as ExpenseCategoryResponse) : null;
  }

  async createExpenseCategory(data: CreateExpenseCategoryDto) {
    const pool = await getPool();

    const query = `
      INSERT INTO expense_categories (category_name, description, default_billable, default_markup_percent, sort_order)
      VALUES (?, ?, ?, ?, ?)
    `;

    const [result] = await pool.query<ResultSetHeader>(query, [
      data.category_name,
      data.description || null,
      data.default_billable,
      data.default_markup_percent,
      data.sort_order,
    ]);

    return this.getExpenseCategoryById(result.insertId);
  }

  async updateExpenseCategory(id: number, data: UpdateExpenseCategoryDto) {
    const pool = await getPool();

    const updates: string[] = [];
    const params: any[] = [];

    if (data.category_name !== undefined) {
      updates.push('category_name = ?');
      params.push(data.category_name);
    }
    if (data.description !== undefined) {
      updates.push('description = ?');
      params.push(data.description);
    }
    if (data.default_billable !== undefined) {
      updates.push('default_billable = ?');
      params.push(data.default_billable);
    }
    if (data.default_markup_percent !== undefined) {
      updates.push('default_markup_percent = ?');
      params.push(data.default_markup_percent);
    }
    if (data.is_active !== undefined) {
      updates.push('is_active = ?');
      params.push(data.is_active);
    }
    if (data.sort_order !== undefined) {
      updates.push('sort_order = ?');
      params.push(data.sort_order);
    }

    if (updates.length === 0) {
      return this.getExpenseCategoryById(id);
    }

    const query = `UPDATE expense_categories SET ${updates.join(', ')} WHERE id = ?`;
    await pool.query(query, [...params, id]);

    return this.getExpenseCategoryById(id);
  }
}

export const expenseRepo = new ExpenseRepository();
