import { z } from 'zod';

// ================================================
// Fee Arrangement DTOs
// ================================================

// Create Fee Arrangement
export const createFeeArrangementSchema = z.object({
  matter_id: z.number().int().positive(),
  fee_type: z.enum(['Hourly', 'Fixed', 'Capped', 'Contingency', 'Blended', 'Custom']),
  base_hourly_rate: z.number().positive().optional(),
  fixed_amount: z.number().positive().optional(),
  cap_amount: z.number().positive().optional(),
  contingency_percent: z.number().min(0).max(100).optional(),
  billing_frequency: z.enum(['Weekly', 'Bi-weekly', 'Monthly', 'Milestone', 'On Completion']).default('Monthly'),
  payment_terms_days: z.number().int().positive().default(30),
  description: z.string().max(5000).optional(),
  terms_document_path: z.string().max(500).optional(),
  effective_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
});

export type CreateFeeArrangementDto = z.infer<typeof createFeeArrangementSchema>;

// Update Fee Arrangement
export const updateFeeArrangementSchema = z.object({
  fee_type: z.enum(['Hourly', 'Fixed', 'Capped', 'Contingency', 'Blended', 'Custom']).optional(),
  base_hourly_rate: z.number().positive().optional(),
  fixed_amount: z.number().positive().optional(),
  cap_amount: z.number().positive().optional(),
  contingency_percent: z.number().min(0).max(100).optional(),
  billing_frequency: z.enum(['Weekly', 'Bi-weekly', 'Monthly', 'Milestone', 'On Completion']).optional(),
  payment_terms_days: z.number().int().positive().optional(),
  description: z.string().max(5000).optional(),
  terms_document_path: z.string().max(500).optional(),
  end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  is_active: z.boolean().optional(),
});

export type UpdateFeeArrangementDto = z.infer<typeof updateFeeArrangementSchema>;

// ================================================
// Rate Card DTOs
// ================================================

// Create Rate Card
export const createRateCardSchema = z.object({
  user_id: z.number().int().positive().optional(),
  role_id: z.number().int().positive().optional(),
  practice_area: z.string().max(100).optional(),
  client_id: z.number().int().positive().optional(),
  hourly_rate: z.number().positive(),
  currency: z.string().length(3).default('GHS'),
  effective_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  is_default: z.boolean().default(false),
  notes: z.string().max(1000).optional(),
});

export type CreateRateCardDto = z.infer<typeof createRateCardSchema>;

// Update Rate Card
export const updateRateCardSchema = z.object({
  hourly_rate: z.number().positive().optional(),
  end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  is_default: z.boolean().optional(),
  notes: z.string().max(1000).optional(),
});

export type UpdateRateCardDto = z.infer<typeof updateRateCardSchema>;

// Query Parameters
export const queryRateCardsSchema = z.object({
  user_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  role_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  practice_area: z.string().optional(),
  client_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  is_default: z.string().optional().transform(val => val === 'true' ? true : val === 'false' ? false : undefined),
});

export type QueryRateCardsDto = z.infer<typeof queryRateCardsSchema>;

// ================================================
// Response Types
// ================================================

export interface FeeArrangementResponse {
  id: number;
  matter_id: number;
  matter_number: string | null;
  matter_title: string;
  fee_type: 'Hourly' | 'Fixed' | 'Capped' | 'Contingency' | 'Blended' | 'Custom';
  base_hourly_rate: number | null;
  fixed_amount: number | null;
  cap_amount: number | null;
  contingency_percent: number | null;
  billing_frequency: 'Weekly' | 'Bi-weekly' | 'Monthly' | 'Milestone' | 'On Completion';
  payment_terms_days: number;
  description: string | null;
  terms_document_path: string | null;
  effective_date: string;
  end_date: string | null;
  is_active: boolean;
  created_by: number;
  created_by_name: string;
  created_at: string;
  updated_at: string;
}

export interface RateCardResponse {
  id: number;
  user_id: number | null;
  user_name: string | null;
  role_id: number | null;
  role_name: string | null;
  practice_area: string | null;
  client_id: number | null;
  client_name: string | null;
  hourly_rate: number;
  currency: string;
  effective_date: string;
  end_date: string | null;
  is_default: boolean;
  notes: string | null;
  created_by: number;
  created_by_name: string;
  created_at: string;
  updated_at: string;
}
