import { Router, Request, Response, NextFunction } from 'express';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { feeRepo } from './repo.js';
import {
  createFeeArrangementSchema,
  updateFeeArrangementSchema,
  createRateCardSchema,
  updateRateCardSchema,
  queryRateCardsSchema,
} from './dto.js';

const router = Router();

// ================================================
// Fee Arrangements
// ================================================

/**
 * GET /api/fees/arrangements
 * List all fee arrangements (optionally filtered by matter)
 */
router.get(
  '/arrangements',
  authenticate,
  requirePermission('matters:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const matterId = req.query.matter_id ? parseInt(req.query.matter_id as string) : undefined;
      const arrangements = await feeRepo.listFeeArrangements(matterId);
      res.json({ success: true, data: { arrangements } });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/fees/arrangements/:id
 * Get single fee arrangement by ID
 */
router.get(
  '/arrangements/:id',
  authenticate,
  requirePermission('matters:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const arrangement = await feeRepo.getFeeArrangementById(id);

      if (!arrangement) {
        res.status(404).json({ success: false, error: 'Fee arrangement not found' });
        return;
      }

      res.json({ success: true, data: arrangement });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/fees/matters/:matterId/arrangement
 * Get active fee arrangement for a matter
 */
router.get(
  '/matters/:matterId/arrangement',
  authenticate,
  requirePermission('matters:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const matterId = parseInt(req.params.matterId);
      const arrangement = await feeRepo.getFeeArrangementByMatter(matterId);

      if (!arrangement) {
        res.json({ success: true, data: null });
        return;
      }

      res.json({ success: true, data: arrangement });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/fees/matters/:matterId/arrangement
 * Create fee arrangement for a matter
 */
router.post(
  '/matters/:matterId/arrangement',
  authenticate,
  requirePermission('matters:update'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const matterId = parseInt(req.params.matterId);
      const data = createFeeArrangementSchema.parse({
        ...req.body,
        matter_id: matterId,
      });
      const arrangement = await feeRepo.createFeeArrangement(data, req.user!.id);
      res.status(201).json({ success: true, data: arrangement });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * PUT /api/fees/arrangements/:id
 * Update fee arrangement
 */
router.put(
  '/arrangements/:id',
  authenticate,
  requirePermission('matters:update'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const data = updateFeeArrangementSchema.parse(req.body);
      const arrangement = await feeRepo.updateFeeArrangement(id, data);
      res.json({ success: true, data: arrangement });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * DELETE /api/fees/arrangements/:id
 * Delete (deactivate) fee arrangement
 */
router.delete(
  '/arrangements/:id',
  authenticate,
  requirePermission('matters:update'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      await feeRepo.deleteFeeArrangement(id);
      res.json({ success: true, message: 'Fee arrangement deactivated successfully' });
    } catch (error) {
      next(error);
    }
  }
);

// ================================================
// Rate Cards
// ================================================

/**
 * GET /api/fees/rate-cards
 * List rate cards with filters
 */
router.get(
  '/rate-cards',
  authenticate,
  requirePermission('matters:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const query = queryRateCardsSchema.parse(req.query);
      const rateCards = await feeRepo.listRateCards(query);
      res.json({ success: true, data: { rateCards } });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/fees/rate-cards/:id
 * Get single rate card by ID
 */
router.get(
  '/rate-cards/:id',
  authenticate,
  requirePermission('matters:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const rateCard = await feeRepo.getRateCardById(id);

      if (!rateCard) {
        res.status(404).json({ success: false, error: 'Rate card not found' });
        return;
      }

      res.json({ success: true, data: rateCard });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/fees/rate-cards
 * Create rate card (admin/manager only)
 */
router.post(
  '/rate-cards',
  authenticate,
  requirePermission('matters:update'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const data = createRateCardSchema.parse(req.body);
      const rateCard = await feeRepo.createRateCard(data, req.user!.id);
      res.status(201).json({ success: true, data: rateCard });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * PUT /api/fees/rate-cards/:id
 * Update rate card (admin/manager only)
 */
router.put(
  '/rate-cards/:id',
  authenticate,
  requirePermission('matters:update'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const data = updateRateCardSchema.parse(req.body);
      const rateCard = await feeRepo.updateRateCard(id, data);
      res.json({ success: true, data: rateCard });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * DELETE /api/fees/rate-cards/:id
 * Delete rate card (admin/manager only)
 */
router.delete(
  '/rate-cards/:id',
  authenticate,
  requirePermission('matters:update'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      await feeRepo.deleteRateCard(id);
      res.json({ success: true, message: 'Rate card deleted successfully' });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/fees/users/:userId/matters/:matterId/rate
 * Get applicable hourly rate for a user on a specific matter
 */
router.get(
  '/users/:userId/matters/:matterId/rate',
  authenticate,
  requirePermission('matters:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const userId = parseInt(req.params.userId);
      const matterId = parseInt(req.params.matterId);
      const rate = await feeRepo.getApplicableRate(userId, matterId);

      res.json({ 
        success: true, 
        data: { 
          hourly_rate: rate,
          user_id: userId,
          matter_id: matterId 
        } 
      });
    } catch (error) {
      next(error);
    }
  }
);

export default router;
