import { z } from 'zod';

// ================================================
// Invoice DTOs
// ================================================

// Create Invoice Line Item
export const createInvoiceLineItemSchema = z.object({
  item_type: z.enum(['Time', 'Expense', 'Fee', 'Adjustment']),
  time_entry_id: z.number().int().positive().optional(),
  expense_id: z.number().int().positive().optional(),
  description: z.string().min(1).max(5000),
  quantity: z.number().positive().default(1),
  unit_price: z.number(),
  amount: z.number(),
  is_taxable: z.boolean().default(true),
});

export type CreateInvoiceLineItemDto = z.infer<typeof createInvoiceLineItemSchema>;

// Create Invoice
export const createInvoiceSchema = z.object({
  matter_id: z.number().int().positive(),
  client_id: z.number().int().positive(),
  invoice_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  due_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  time_entry_ids: z.array(z.number().int().positive()).optional(),
  expense_ids: z.array(z.number().int().positive()).optional(),
  line_items: z.array(createInvoiceLineItemSchema).optional(),
  notes: z.string().max(5000).optional(),
  payment_instructions: z.string().max(2000).optional(),
  apply_vat: z.boolean().default(true),
  apply_nhil: z.boolean().default(true),
  apply_getfund: z.boolean().default(true),
  apply_wht: z.boolean().default(false),
  wht_rate: z.number().min(0).max(100).default(7.5),
});

export type CreateInvoiceDto = z.infer<typeof createInvoiceSchema>;

// Update Invoice
export const updateInvoiceSchema = z.object({
  invoice_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  due_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  notes: z.string().max(5000).optional(),
  payment_instructions: z.string().max(2000).optional(),
  status: z.enum(['Draft', 'Sent', 'Paid', 'Partially Paid', 'Overdue', 'Cancelled']).optional(),
});

export type UpdateInvoiceDto = z.infer<typeof updateInvoiceSchema>;

// Query Parameters
export const queryInvoicesSchema = z.object({
  page: z.string().optional().transform(val => val ? parseInt(val) : 1),
  limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
  matter_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  client_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  status: z.enum(['Draft', 'Sent', 'Paid', 'Partially Paid', 'Overdue', 'Cancelled']).optional(),
  start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  overdue_only: z.string().optional().transform(val => val === 'true'),
});

export type QueryInvoicesDto = z.infer<typeof queryInvoicesSchema>;

// Send Invoice
export const sendInvoiceSchema = z.object({
  send_email: z.boolean().default(true),
  email_message: z.string().max(2000).optional(),
});

export type SendInvoiceDto = z.infer<typeof sendInvoiceSchema>;

// Record Payment
export const recordPaymentSchema = z.object({
  amount: z.number().positive(),
  payment_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  payment_method: z.string().max(50).optional(),
  reference_number: z.string().max(100).optional(),
  notes: z.string().max(1000).optional(),
});

export type RecordPaymentDto = z.infer<typeof recordPaymentSchema>;

// ================================================
// Response Types
// ================================================

export interface InvoiceLineItemResponse {
  id: number;
  invoice_id: number;
  item_type: 'Time' | 'Expense' | 'Fee' | 'Adjustment';
  time_entry_id: number | null;
  expense_id: number | null;
  description: string;
  quantity: number;
  unit_price: number;
  amount: number;
  is_taxable: boolean;
  created_at: string;
}

export interface InvoiceResponse {
  id: number;
  invoice_number: string;
  matter_id: number;
  matter_number: string | null;
  matter_title: string;
  client_id: number;
  client_name: string;
  client_email: string | null;
  invoice_date: string;
  due_date: string;
  subtotal: number;
  vat_amount: number;
  nhil_amount: number;
  getfund_amount: number;
  total_before_wht: number;
  wht_amount: number;
  total_amount: number;
  amount_paid: number;
  balance_due: number;
  status: 'Draft' | 'Sent' | 'Paid' | 'Partially Paid' | 'Overdue' | 'Cancelled';
  notes: string | null;
  payment_instructions: string | null;
  sent_at: string | null;
  sent_by: number | null;
  sent_by_name: string | null;
  created_by: number;
  created_by_name: string;
  created_at: string;
  updated_at: string;
  line_items?: InvoiceLineItemResponse[];
}

export interface InvoiceSummaryResponse {
  total_invoices: number;
  total_amount: number;
  total_paid: number;
  total_outstanding: number;
  draft_count: number;
  sent_count: number;
  overdue_count: number;
  paid_count: number;
}

export interface PaymentRecordResponse {
  id: number;
  invoice_id: number;
  invoice_number: string;
  amount: number;
  payment_date: string;
  payment_method: string | null;
  reference_number: string | null;
  notes: string | null;
  recorded_by: number;
  recorded_by_name: string;
  created_at: string;
}
