import { RowDataPacket, ResultSetHeader } from 'mysql2/promise';
import { getPool } from '../../db/pool.js';
import type {
  CreateInvoiceDto,
  UpdateInvoiceDto,
  QueryInvoicesDto,
  RecordPaymentDto,
  InvoiceResponse,
  InvoiceLineItemResponse,
  InvoiceSummaryResponse,
  PaymentRecordResponse,
} from './dto.js';

// Ghana Tax Rates (as of 2025)
const TAX_RATES = {
  VAT: 0.15,        // 15%
  NHIL: 0.025,      // 2.5%
  GETFUND: 0.025,   // 2.5%
  WHT_DEFAULT: 0.075, // 7.5% for services
};

export class InvoiceRepository {
  // ================================================
  // Invoice Management
  // ================================================

  async listInvoices(query: QueryInvoicesDto) {
    const pool = await getPool();
    const { page = 1, limit = 50, matter_id, client_id, status, start_date, end_date, overdue_only } = query;
    const offset = (page - 1) * limit;

    let whereConditions: string[] = [];
    let params: any[] = [];

    if (matter_id) {
      whereConditions.push('i.matter_id = ?');
      params.push(matter_id);
    }

    if (client_id) {
      whereConditions.push('i.client_id = ?');
      params.push(client_id);
    }

    if (status) {
      whereConditions.push('i.status = ?');
      params.push(status);
    }

    if (start_date) {
      whereConditions.push('i.invoice_date >= ?');
      params.push(start_date);
    }

    if (end_date) {
      whereConditions.push('i.invoice_date <= ?');
      params.push(end_date);
    }

    if (overdue_only) {
      whereConditions.push('i.status = "Sent" AND i.due_date < CURDATE() AND i.balance_due > 0');
    }

    const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';

    // Get total count
    const countQuery = `SELECT COUNT(*) as total FROM invoices i ${whereClause}`;
    const [countRows] = await pool.query<RowDataPacket[]>(countQuery, params);
    const total = countRows[0].total;

    // Get invoices
    const query_sql = `
      SELECT 
        i.*,
        m.matter_number,
        m.title as matter_title,
        c.name as client_name,
        c.email as client_email,
        creator.username as created_by_name,
        sender.username as sent_by_name
      FROM invoices i
      JOIN matters m ON i.matter_id = m.id
      JOIN clients c ON i.client_id = c.id
      JOIN users creator ON i.created_by = creator.id
      LEFT JOIN users sender ON i.sent_by = sender.id
      ${whereClause}
      ORDER BY i.invoice_date DESC, i.created_at DESC
      LIMIT ? OFFSET ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query_sql, [...params, limit, offset]);

    return {
      invoices: rows as InvoiceResponse[],
      pagination: {
        page,
        limit,
        total,
        totalPages: Math.ceil(total / limit),
      },
    };
  }

  async getInvoiceById(id: number, includeLineItems: boolean = true): Promise<InvoiceResponse | null> {
    const pool = await getPool();

    const query = `
      SELECT 
        i.*,
        m.matter_number,
        m.title as matter_title,
        c.name as client_name,
        c.email as client_email,
        creator.username as created_by_name,
        sender.username as sent_by_name
      FROM invoices i
      JOIN matters m ON i.matter_id = m.id
      JOIN clients c ON i.client_id = c.id
      JOIN users creator ON i.created_by = creator.id
      LEFT JOIN users sender ON i.sent_by = sender.id
      WHERE i.id = ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, [id]);

    if (rows.length === 0) {
      return null;
    }

    const invoice = rows[0] as InvoiceResponse;

    if (includeLineItems) {
      invoice.line_items = await this.getInvoiceLineItems(id);
    }

    return invoice;
  }

  async getInvoiceLineItems(invoiceId: number): Promise<InvoiceLineItemResponse[]> {
    const pool = await getPool();

    const query = `
      SELECT * FROM invoice_line_items
      WHERE invoice_id = ?
      ORDER BY id
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, [invoiceId]);
    return rows as InvoiceLineItemResponse[];
  }

  async generateInvoiceNumber(year: number): Promise<string> {
    const pool = await getPool();

    // Get the last invoice number for this year
    const query = `
      SELECT invoice_number 
      FROM invoices 
      WHERE invoice_number LIKE ?
      ORDER BY invoice_number DESC 
      LIMIT 1
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, [`INV-${year}-%`]);

    let nextNumber = 1;
    if (rows.length > 0) {
      const lastNumber = rows[0].invoice_number;
      const parts = lastNumber.split('-');
      if (parts.length === 3) {
        nextNumber = parseInt(parts[2]) + 1;
      }
    }

    return `INV-${year}-${nextNumber.toString().padStart(5, '0')}`;
  }

  async createInvoice(data: CreateInvoiceDto, userId: number): Promise<InvoiceResponse | null> {
    const pool = await getPool();
    const connection = await pool.getConnection();

    try {
      await connection.beginTransaction();

      // Generate invoice number
      const invoiceDate = new Date(data.invoice_date);
      const invoiceNumber = await this.generateInvoiceNumber(invoiceDate.getFullYear());

      // Collect line items from time entries
      const lineItems: any[] = [];

      if (data.time_entry_ids && data.time_entry_ids.length > 0) {
        const timeQuery = `
          SELECT 
            te.id,
            te.description,
            te.hours as quantity,
            te.hourly_rate as unit_price,
            te.total_amount as amount,
            te.billable
          FROM time_entries te
          WHERE te.id IN (${data.time_entry_ids.map(() => '?').join(',')})
            AND te.status = 'Approved'
            AND te.billable = TRUE
        `;

        const [timeEntries] = await connection.query<RowDataPacket[]>(timeQuery, data.time_entry_ids);

        for (const entry of timeEntries) {
          lineItems.push({
            item_type: 'Time',
            time_entry_id: entry.id,
            description: entry.description,
            quantity: entry.quantity,
            unit_price: entry.unit_price || 0,
            amount: entry.amount || 0,
            is_taxable: true,
          });
        }
      }

      // Collect line items from expenses
      if (data.expense_ids && data.expense_ids.length > 0) {
        const expenseQuery = `
          SELECT 
            ee.id,
            ee.description,
            1 as quantity,
            ee.billable_amount as unit_price,
            ee.billable_amount as amount,
            ee.billable
          FROM expense_entries ee
          WHERE ee.id IN (${data.expense_ids.map(() => '?').join(',')})
            AND ee.status = 'Approved'
            AND ee.billable = TRUE
        `;

        const [expenses] = await connection.query<RowDataPacket[]>(expenseQuery, data.expense_ids);

        for (const expense of expenses) {
          lineItems.push({
            item_type: 'Expense',
            expense_id: expense.id,
            description: expense.description,
            quantity: expense.quantity,
            unit_price: expense.unit_price || 0,
            amount: expense.amount || 0,
            is_taxable: true,
          });
        }
      }

      // Add custom line items
      if (data.line_items && data.line_items.length > 0) {
        lineItems.push(...data.line_items);
      }

      if (lineItems.length === 0) {
        throw new Error('Invoice must have at least one line item');
      }

      // Calculate totals
      const subtotal = lineItems.reduce((sum, item) => sum + item.amount, 0);
      const taxableAmount = lineItems
        .filter(item => item.is_taxable)
        .reduce((sum, item) => sum + item.amount, 0);

      const vat_amount = data.apply_vat ? taxableAmount * TAX_RATES.VAT : 0;
      const nhil_amount = data.apply_nhil ? taxableAmount * TAX_RATES.NHIL : 0;
      const getfund_amount = data.apply_getfund ? taxableAmount * TAX_RATES.GETFUND : 0;
      const total_before_wht = subtotal + vat_amount + nhil_amount + getfund_amount;
      const wht_amount = data.apply_wht ? subtotal * (data.wht_rate / 100) : 0;
      const total_amount = total_before_wht - wht_amount;

      // Create invoice
      const invoiceQuery = `
        INSERT INTO invoices (
          invoice_number, matter_id, client_id, invoice_date, due_date,
          subtotal, vat_amount, nhil_amount, getfund_amount,
          total_before_wht, wht_amount, total_amount,
          amount_paid, balance_due, status,
          notes, payment_instructions, created_by
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `;

      const [result] = await connection.query<ResultSetHeader>(invoiceQuery, [
        invoiceNumber,
        data.matter_id,
        data.client_id,
        data.invoice_date,
        data.due_date,
        subtotal,
        vat_amount,
        nhil_amount,
        getfund_amount,
        total_before_wht,
        wht_amount,
        total_amount,
        0, // amount_paid
        total_amount, // balance_due
        'Draft',
        data.notes || null,
        data.payment_instructions || null,
        userId,
      ]);

      const invoiceId = result.insertId;

      // Insert line items
      for (const item of lineItems) {
        const lineItemQuery = `
          INSERT INTO invoice_line_items (
            invoice_id, item_type, time_entry_id, expense_id,
            description, quantity, unit_price, amount, is_taxable
          ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        `;

        await connection.query(lineItemQuery, [
          invoiceId,
          item.item_type,
          item.time_entry_id || null,
          item.expense_id || null,
          item.description,
          item.quantity,
          item.unit_price,
          item.amount,
          item.is_taxable,
        ]);
      }

      // Mark time entries as invoiced
      if (data.time_entry_ids && data.time_entry_ids.length > 0) {
        await connection.query(
          `UPDATE time_entries SET status = 'Invoiced' WHERE id IN (${data.time_entry_ids.map(() => '?').join(',')})`,
          data.time_entry_ids
        );
      }

      // Mark expenses as invoiced
      if (data.expense_ids && data.expense_ids.length > 0) {
        await connection.query(
          `UPDATE expense_entries SET status = 'Invoiced' WHERE id IN (${data.expense_ids.map(() => '?').join(',')})`,
          data.expense_ids
        );
      }

      await connection.commit();
      connection.release();

      return this.getInvoiceById(invoiceId);
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  }

  async updateInvoice(id: number, data: UpdateInvoiceDto): Promise<InvoiceResponse | null> {
    const pool = await getPool();

    const existing = await this.getInvoiceById(id, false);
    if (!existing) {
      throw new Error('Invoice not found');
    }

    // Only allow updates if status is Draft
    if (existing.status !== 'Draft' && !data.status) {
      throw new Error('Cannot update invoice that has been sent (except status)');
    }

    const updates: string[] = [];
    const params: any[] = [];

    if (data.invoice_date !== undefined) {
      updates.push('invoice_date = ?');
      params.push(data.invoice_date);
    }
    if (data.due_date !== undefined) {
      updates.push('due_date = ?');
      params.push(data.due_date);
    }
    if (data.notes !== undefined) {
      updates.push('notes = ?');
      params.push(data.notes);
    }
    if (data.payment_instructions !== undefined) {
      updates.push('payment_instructions = ?');
      params.push(data.payment_instructions);
    }
    if (data.status !== undefined) {
      updates.push('status = ?');
      params.push(data.status);
    }

    if (updates.length === 0) {
      return existing;
    }

    const query = `UPDATE invoices SET ${updates.join(', ')} WHERE id = ?`;
    await pool.query(query, [...params, id]);

    return this.getInvoiceById(id);
  }

  async sendInvoice(id: number, userId: number): Promise<InvoiceResponse | null> {
    const pool = await getPool();

    const existing = await this.getInvoiceById(id, false);
    if (!existing) {
      throw new Error('Invoice not found');
    }

    if (existing.status !== 'Draft') {
      throw new Error('Only draft invoices can be sent');
    }

    await pool.query(
      'UPDATE invoices SET status = "Sent", sent_at = NOW(), sent_by = ? WHERE id = ?',
      [userId, id]
    );

    return this.getInvoiceById(id);
  }

  async recordPayment(invoiceId: number, data: RecordPaymentDto, userId: number): Promise<PaymentRecordResponse | null> {
    const pool = await getPool();
    const connection = await pool.getConnection();

    try {
      await connection.beginTransaction();

      const invoice = await this.getInvoiceById(invoiceId, false);
      if (!invoice) {
        throw new Error('Invoice not found');
      }

      if (invoice.status === 'Cancelled') {
        throw new Error('Cannot record payment for cancelled invoice');
      }

      if (data.amount > invoice.balance_due) {
        throw new Error('Payment amount exceeds balance due');
      }

      // Record payment (Note: payments table doesn't exist in schema, using comment for future)
      // For now, we'll update the invoice directly
      const newAmountPaid = invoice.amount_paid + data.amount;
      const newBalanceDue = invoice.balance_due - data.amount;
      let newStatus = invoice.status;

      if (newBalanceDue === 0) {
        newStatus = 'Paid';
      } else if (newAmountPaid > 0) {
        newStatus = 'Partially Paid';
      }

      await connection.query(
        'UPDATE invoices SET amount_paid = ?, balance_due = ?, status = ? WHERE id = ?',
        [newAmountPaid, newBalanceDue, newStatus, invoiceId]
      );

      await connection.commit();
      connection.release();

      // Return payment record (simplified for now)
      return {
        id: invoiceId,
        invoice_id: invoiceId,
        invoice_number: invoice.invoice_number,
        amount: data.amount,
        payment_date: data.payment_date,
        payment_method: data.payment_method || null,
        reference_number: data.reference_number || null,
        notes: data.notes || null,
        recorded_by: userId,
        recorded_by_name: '', // Will be filled by route
        created_at: new Date().toISOString(),
      };
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  }

  async getInvoiceSummary(clientId?: number, matterId?: number): Promise<InvoiceSummaryResponse> {
    const pool = await getPool();

    let whereConditions: string[] = [];
    let params: any[] = [];

    if (clientId) {
      whereConditions.push('client_id = ?');
      params.push(clientId);
    }

    if (matterId) {
      whereConditions.push('matter_id = ?');
      params.push(matterId);
    }

    const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';

    const query = `
      SELECT 
        COUNT(*) as total_invoices,
        COALESCE(SUM(total_amount), 0) as total_amount,
        COALESCE(SUM(amount_paid), 0) as total_paid,
        COALESCE(SUM(balance_due), 0) as total_outstanding,
        SUM(CASE WHEN status = 'Draft' THEN 1 ELSE 0 END) as draft_count,
        SUM(CASE WHEN status = 'Sent' THEN 1 ELSE 0 END) as sent_count,
        SUM(CASE WHEN status = 'Sent' AND due_date < CURDATE() AND balance_due > 0 THEN 1 ELSE 0 END) as overdue_count,
        SUM(CASE WHEN status = 'Paid' THEN 1 ELSE 0 END) as paid_count
      FROM invoices
      ${whereClause}
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, params);
    return rows[0] as InvoiceSummaryResponse;
  }

  async cancelInvoice(id: number): Promise<void> {
    const pool = await getPool();

    const existing = await this.getInvoiceById(id, false);
    if (!existing) {
      throw new Error('Invoice not found');
    }

    if (existing.status === 'Paid' || existing.status === 'Partially Paid') {
      throw new Error('Cannot cancel invoice that has received payments');
    }

    await pool.query('UPDATE invoices SET status = "Cancelled" WHERE id = ?', [id]);
  }
}

export const invoiceRepo = new InvoiceRepository();
