import { Router, Request, Response, NextFunction } from 'express';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { invoiceRepo } from './repo.js';
import {
  createInvoiceSchema,
  updateInvoiceSchema,
  queryInvoicesSchema,
  sendInvoiceSchema,
  recordPaymentSchema,
} from './dto.js';

const router = Router();

// ================================================
// Invoice Management
// ================================================

/**
 * GET /api/invoices
 * List invoices with filters
 */
router.get(
  '/',
  authenticate,
  requirePermission('invoice:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const query = queryInvoicesSchema.parse(req.query);
      const result = await invoiceRepo.listInvoices(query);
      res.json({ success: true, data: result });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/invoices/summary
 * Get invoice summary statistics
 */
router.get(
  '/summary',
  authenticate,
  requirePermission('invoice:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const clientId = req.query.client_id ? parseInt(req.query.client_id as string) : undefined;
      const matterId = req.query.matter_id ? parseInt(req.query.matter_id as string) : undefined;
      const summary = await invoiceRepo.getInvoiceSummary(clientId, matterId);
      res.json({ success: true, data: summary });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/invoices/:id
 * Get single invoice by ID (with line items)
 */
router.get(
  '/:id',
  authenticate,
  requirePermission('invoice:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const invoice = await invoiceRepo.getInvoiceById(id, true);

      if (!invoice) {
        res.status(404).json({ success: false, error: 'Invoice not found' });
        return;
      }

      res.json({ success: true, data: invoice });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/invoices
 * Create new invoice with line items
 */
router.post(
  '/',
  authenticate,
  requirePermission('invoice:create'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const data = createInvoiceSchema.parse(req.body);
      const invoice = await invoiceRepo.createInvoice(data, req.user!.id);
      res.status(201).json({ success: true, data: invoice });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * PUT /api/invoices/:id
 * Update invoice (Draft invoices only, or status changes)
 */
router.put(
  '/:id',
  authenticate,
  requirePermission('invoice:update'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const data = updateInvoiceSchema.parse(req.body);
      const invoice = await invoiceRepo.updateInvoice(id, data);
      res.json({ success: true, data: invoice });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/invoices/:id/send
 * Send invoice to client (mark as Sent)
 */
router.post(
  '/:id/send',
  authenticate,
  requirePermission('invoice:send'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      sendInvoiceSchema.parse(req.body); // Validate but don't use for now
      const invoice = await invoiceRepo.sendInvoice(id, req.user!.id);
      res.json({ 
        success: true, 
        data: invoice,
        message: 'Invoice sent successfully' 
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/invoices/:id/cancel
 * Cancel invoice (no payments received)
 */
router.post(
  '/:id/cancel',
  authenticate,
  requirePermission('invoice:update'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      await invoiceRepo.cancelInvoice(id);
      res.json({ 
        success: true, 
        message: 'Invoice cancelled successfully' 
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/invoices/:id/payment
 * Record payment for invoice
 */
router.post(
  '/:id/payment',
  authenticate,
  requirePermission('invoice:payment'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const data = recordPaymentSchema.parse(req.body);
      const payment = await invoiceRepo.recordPayment(id, data, req.user!.id);
      
      res.json({ 
        success: true, 
        data: payment,
        message: 'Payment recorded successfully' 
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/invoices/:id/line-items
 * Get invoice line items
 */
router.get(
  '/:id/line-items',
  authenticate,
  requirePermission('invoice:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const lineItems = await invoiceRepo.getInvoiceLineItems(id);
      res.json({ success: true, data: { lineItems } });
    } catch (error) {
      next(error);
    }
  }
);

export default router;
