import { getPool } from '../../db/pool.js';
import { 
  CreateMatterDto, 
  UpdateMatterDto, 
  AddPartyDto, 
  MatterResponse, 
  ConflictPartyResponse,
  MatterWithParties,
  ListMattersQuery 
} from './dto.js';

export class MatterRepository {
  /**
   * Get all matters with client and lawyer info (paginated)
   */
  async findAll(query: ListMattersQuery): Promise<{ matters: MatterResponse[]; total: number }> {
    const pool = await getPool();
    const { page = 1, limit = 20, case_type, case_status, client_id, lawyer_id, search } = query;
    const offset = (page - 1) * limit;

    let whereConditions: string[] = ['c.is_active = 1'];
    const params: any[] = [];

    if (case_type) {
      whereConditions.push('c.case_type = ?');
      params.push(case_type);
    }

    if (case_status) {
      whereConditions.push('c.case_status = ?');
      params.push(case_status);
    }

    if (client_id) {
      whereConditions.push('c.client_id = ?');
      params.push(client_id);
    }

    if (lawyer_id) {
      whereConditions.push('c.lawyer_id = ?');
      params.push(lawyer_id);
    }

    if (search) {
      whereConditions.push('(c.matter_id LIKE ? OR c.case_number LIKE ? OR c.title LIKE ?)');
      const searchPattern = `%${search}%`;
      params.push(searchPattern, searchPattern, searchPattern);
    }

    const whereClause = whereConditions.length > 0 ? `WHERE ${whereConditions.join(' AND ')}` : '';

    // Get total count
    const [countResult] = await pool.query<any[]>(
      `SELECT COUNT(*) as total FROM cases c ${whereClause}`,
      params
    );
    const total = countResult[0].total;

    // Get paginated results
    const [rows] = await pool.query<any[]>(
      `SELECT 
        c.id, c.matter_id, c.case_number, c.case_type, c.case_status,
        c.title, c.description, c.filing_date, c.closing_date,
        c.location, c.latitude, c.longitude,
        c.client_id, cl.name as client_name,
        c.lawyer_id, CONCAT(e.first_name, ' ', e.surname) as lawyer_name,
        c.workflow_id, w.workflow_name,
        c.current_stage, c.is_active,
        c.created_at, c.updated_at
       FROM cases c
       JOIN clients cl ON cl.id = c.client_id
       JOIN lawyers l ON l.id = c.lawyer_id
       JOIN employees e ON e.id = l.employee_id
       LEFT JOIN workflow_config w ON w.id = c.workflow_id
       ${whereClause}
       ORDER BY c.created_at DESC
       LIMIT ? OFFSET ?`,
      [...params, limit, offset]
    );

    return { matters: rows, total };
  }

  /**
   * Find matter by ID with full details
   */
  async findById(id: number): Promise<MatterWithParties | null> {
    const pool = await getPool();

    const [rows] = await pool.query<any[]>(
      `SELECT 
        c.id, c.matter_id, c.case_number, c.case_type, c.case_status,
        c.title, c.description, c.filing_date, c.closing_date,
        c.location, c.latitude, c.longitude,
        c.client_id, cl.name as client_name,
        c.lawyer_id, CONCAT(e.first_name, ' ', e.surname) as lawyer_name,
        c.workflow_id, w.workflow_name,
        c.current_stage, c.is_active,
        c.created_at, c.updated_at
       FROM cases c
       JOIN clients cl ON cl.id = c.client_id
       JOIN lawyers l ON l.id = c.lawyer_id
       JOIN employees e ON e.id = l.employee_id
       LEFT JOIN workflow_config w ON w.id = c.workflow_id
       WHERE c.id = ?`,
      [id]
    );

    if (rows.length === 0) return null;

    const matter = rows[0];

    // Get parties
    const parties = await this.getPartiesByMatterId(id);

    return { ...matter, parties };
  }

  /**
   * Create a new matter
   */
  async create(data: CreateMatterDto, matter_id: string): Promise<number> {
    const pool = await getPool();

    const [result] = await pool.query<any>(
      `INSERT INTO cases (
        matter_id, case_number, case_type, case_status, title, description,
        client_id, lawyer_id, filing_date, location, latitude, longitude,
        workflow_id, current_stage
      ) VALUES (?, ?, ?, 'Active', ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        matter_id,
        data.case_number || matter_id, // Use matter_id as case_number if not provided
        data.case_type,
        data.title,
        data.description || null,
        data.client_id,
        data.lawyer_id,
        data.filing_date,
        data.location || null,
        data.latitude || null,
        data.longitude || null,
        data.workflow_id || null,
        data.workflow_id ? 'Intake' : null, // Set initial stage if workflow assigned
      ]
    );

    return result.insertId;
  }

  /**
   * Update matter
   */
  async update(id: number, data: UpdateMatterDto): Promise<boolean> {
    const pool = await getPool();

    const fields: string[] = [];
    const values: any[] = [];

    if (data.case_number !== undefined) {
      fields.push('case_number = ?');
      values.push(data.case_number);
    }
    if (data.case_type !== undefined) {
      fields.push('case_type = ?');
      values.push(data.case_type);
    }
    if (data.case_status !== undefined) {
      fields.push('case_status = ?');
      values.push(data.case_status);
    }
    if (data.title !== undefined) {
      fields.push('title = ?');
      values.push(data.title);
    }
    if (data.description !== undefined) {
      fields.push('description = ?');
      values.push(data.description);
    }
    if (data.client_id !== undefined) {
      fields.push('client_id = ?');
      values.push(data.client_id);
    }
    if (data.lawyer_id !== undefined) {
      fields.push('lawyer_id = ?');
      values.push(data.lawyer_id);
    }
    if (data.location !== undefined) {
      fields.push('location = ?');
      values.push(data.location);
    }
    if (data.latitude !== undefined) {
      fields.push('latitude = ?');
      values.push(data.latitude);
    }
    if (data.longitude !== undefined) {
      fields.push('longitude = ?');
      values.push(data.longitude);
    }
    if (data.workflow_id !== undefined) {
      fields.push('workflow_id = ?');
      values.push(data.workflow_id);
    }
    if (data.current_stage !== undefined) {
      fields.push('current_stage = ?');
      values.push(data.current_stage);
    }

    if (fields.length === 0) return false;

    values.push(id);

    const [result] = await pool.query<any>(
      `UPDATE cases SET ${fields.join(', ')} WHERE id = ?`,
      values
    );

    return result.affectedRows > 0;
  }

  /**
   * Soft delete matter
   */
  async delete(id: number): Promise<boolean> {
    const pool = await getPool();

    const [result] = await pool.query<any>(
      `UPDATE cases SET is_active = 0 WHERE id = ?`,
      [id]
    );

    return result.affectedRows > 0;
  }

  /**
   * Add party to matter for conflict tracking
   */
  async addParty(matter_id: number, data: AddPartyDto): Promise<number> {
    const pool = await getPool();

    const [result] = await pool.query<any>(
      `INSERT INTO conflict_parties (matter_id, party_type, name, identifier, relationship)
       VALUES (?, ?, ?, ?, ?)`,
      [matter_id, data.party_type, data.name, data.identifier || null, data.relationship || null]
    );

    return result.insertId;
  }

  /**
   * Get parties by matter ID
   */
  async getPartiesByMatterId(matter_id: number): Promise<ConflictPartyResponse[]> {
    const pool = await getPool();

    const [rows] = await pool.query<any[]>(
      `SELECT id, matter_id, party_type, name, identifier, relationship, created_at
       FROM conflict_parties
       WHERE matter_id = ?
       ORDER BY created_at DESC`,
      [matter_id]
    );

    return rows;
  }

  /**
   * Get next sequence number for matter ID generation
   */
  async getNextSequence(): Promise<{ current: number; config: any }> {
    const pool = await getPool();

    const [config] = await pool.query<any[]>(
      `SELECT * FROM matter_id_config WHERE is_active = 1 LIMIT 1`
    );

    if (config.length === 0) {
      throw new Error('No active matter ID configuration found');
    }

    const cfg = config[0];

    // Check if we need to reset (new year)
    const currentYear = new Date().getFullYear();
    if (cfg.reset_yearly && cfg.last_reset_year !== currentYear) {
      // Reset sequence
      await pool.query(
        `UPDATE matter_id_config 
         SET current_sequence = ?, last_reset_year = ? 
         WHERE id = ?`,
        [cfg.sequence_start, currentYear, cfg.id]
      );
      return { current: cfg.sequence_start, config: { ...cfg, current_sequence: cfg.sequence_start } };
    }

    // Increment sequence
    const nextSeq = cfg.current_sequence + 1;
    await pool.query(
      `UPDATE matter_id_config SET current_sequence = ? WHERE id = ?`,
      [nextSeq, cfg.id]
    );

    return { current: nextSeq, config: cfg };
  }
}
