import { Router, Request, Response, NextFunction } from 'express';
import { MatterRepository } from './repo.js';
import { MatterService } from './service.js';
import {
  createMatterSchema,
  updateMatterSchema,
  getMatterSchema,
  deleteMatterSchema,
  addPartySchema,
  listMattersSchema,
} from './dto.js';
import { validate } from '../../middleware/validate.js';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { logger } from '../../utils/logger.js';

const router = Router();
const repo = new MatterRepository();
const service = new MatterService();

// All routes require authentication
router.use(authenticate);

/**
 * GET /api/matters
 * List all matters with pagination and filters
 */
router.get(
  '/',
  requirePermission('matters:read'),
  validate(listMattersSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const query = req.query as any;
      const { matters, total } = await repo.findAll(query);

      const page = parseInt(query.page || '1');
      const limit = parseInt(query.limit || '20');
      const totalPages = Math.ceil(total / limit);

      res.json({
        success: true,
        data: {
          matters,
          pagination: {
            page,
            limit,
            total,
            totalPages,
          },
        },
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id }, 'Failed to list matters');
      next(error);
    }
  }
);

/**
 * GET /api/matters/:id
 * Get matter by ID with parties
 */
router.get(
  '/:id',
  requirePermission('matters:read'),
  validate(getMatterSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { id } = req.params;
      const matter = await repo.findById(parseInt(id));

      if (!matter) {
        return res.status(404).json({
          success: false,
          error: 'Matter not found',
        });
      }

      res.json({
        success: true,
        data: matter,
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id, matterId: req.params.id }, 'Failed to get matter');
      next(error);
    }
  }
);

/**
 * POST /api/matters
 * Create a new matter with auto-generated matter ID
 */
router.post(
  '/',
  requirePermission('matters:create'),
  validate(createMatterSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const data = req.body;

      // Get department from lawyer
      const departmentId = await service.getDepartmentIdFromLawyer(data.lawyer_id);

      // Generate matter ID
      const matter_id = await service.generateMatterId(departmentId || undefined);

      // Create matter
      const matterId = await repo.create(data, matter_id);

      // Add parties if provided
      if (data.parties && data.parties.length > 0) {
        for (const party of data.parties) {
          await repo.addParty(matterId, party);
        }
      }

      // Log audit
      logger.info({
        userId: req.user?.id,
        action: 'CREATE_MATTER',
        matterId,
        matter_id,
      }, 'Matter created');

      // Fetch created matter
      const matter = await repo.findById(matterId);

      res.status(201).json({
        success: true,
        message: 'Matter created successfully',
        data: matter,
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id }, 'Failed to create matter');
      next(error);
    }
  }
);

/**
 * PUT /api/matters/:id
 * Update matter
 */
router.put(
  '/:id',
  requirePermission('matters:update'),
  validate(updateMatterSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { id } = req.params;
      const data = req.body;

      const updated = await repo.update(parseInt(id), data);

      if (!updated) {
        return res.status(404).json({
          success: false,
          error: 'Matter not found or no changes made',
        });
      }

      // Log audit
      logger.info({
        userId: req.user?.id,
        action: 'UPDATE_MATTER',
        matterId: id,
        changes: data,
      }, 'Matter updated');

      // Fetch updated matter
      const matter = await repo.findById(parseInt(id));

      res.json({
        success: true,
        message: 'Matter updated successfully',
        data: matter,
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id, matterId: req.params.id }, 'Failed to update matter');
      next(error);
    }
  }
);

/**
 * DELETE /api/matters/:id
 * Soft delete matter
 */
router.delete(
  '/:id',
  requirePermission('matters:delete'),
  validate(deleteMatterSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { id } = req.params;

      const deleted = await repo.delete(parseInt(id));

      if (!deleted) {
        return res.status(404).json({
          success: false,
          error: 'Matter not found',
        });
      }

      // Log audit
      logger.info({
        userId: req.user?.id,
        action: 'DELETE_MATTER',
        matterId: id,
      }, 'Matter deleted');

      res.json({
        success: true,
        message: 'Matter deleted successfully',
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id, matterId: req.params.id }, 'Failed to delete matter');
      next(error);
    }
  }
);

/**
 * POST /api/matters/:id/parties
 * Add party to matter for conflict tracking
 */
router.post(
  '/:id/parties',
  requirePermission('matters:update'),
  validate(addPartySchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { id } = req.params;
      const data = req.body;

      // Check if matter exists
      const matter = await repo.findById(parseInt(id));
      if (!matter) {
        return res.status(404).json({
          success: false,
          error: 'Matter not found',
        });
      }

      const partyId = await repo.addParty(parseInt(id), data);

      // Log audit
      logger.info({
        userId: req.user?.id,
        action: 'ADD_PARTY',
        matterId: id,
        partyId,
        partyName: data.name,
      }, 'Party added to matter');

      res.status(201).json({
        success: true,
        message: 'Party added successfully',
        data: { id: partyId },
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id, matterId: req.params.id }, 'Failed to add party');
      next(error);
    }
  }
);

/**
 * GET /api/matters/:id/parties
 * Get all parties for a matter
 */
router.get(
  '/:id/parties',
  requirePermission('matters:read'),
  validate(getMatterSchema),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { id } = req.params;

      const parties = await repo.getPartiesByMatterId(parseInt(id));

      res.json({
        success: true,
        data: { parties },
      });
    } catch (error) {
      logger.error({ error, userId: req.user?.id, matterId: req.params.id }, 'Failed to get parties');
      next(error);
    }
  }
);

export default router;
