import { z } from 'zod';

// ================================================
// Receipt DTOs
// ================================================

// Payment Allocation Item
export const paymentAllocationSchema = z.object({
  invoice_id: z.number().int().positive(),
  amount: z.number().positive(),
});

export type PaymentAllocationDto = z.infer<typeof paymentAllocationSchema>;

// Create Receipt
export const createReceiptSchema = z.object({
  client_id: z.number().int().positive(),
  receipt_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  amount: z.number().positive(),
  payment_method: z.enum(['Cash', 'Cheque', 'Bank Transfer', 'Mobile Money', 'Card', 'Other']),
  reference_number: z.string().max(100).optional(),
  bank_name: z.string().max(100).optional(),
  cheque_number: z.string().max(50).optional(),
  notes: z.string().max(2000).optional(),
  allocations: z.array(paymentAllocationSchema).min(1),
});

export type CreateReceiptDto = z.infer<typeof createReceiptSchema>;

// Update Receipt
export const updateReceiptSchema = z.object({
  receipt_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  payment_method: z.enum(['Cash', 'Cheque', 'Bank Transfer', 'Mobile Money', 'Card', 'Other']).optional(),
  reference_number: z.string().max(100).optional(),
  bank_name: z.string().max(100).optional(),
  cheque_number: z.string().max(50).optional(),
  notes: z.string().max(2000).optional(),
  status: z.enum(['Pending', 'Cleared', 'Bounced', 'Cancelled']).optional(),
});

export type UpdateReceiptDto = z.infer<typeof updateReceiptSchema>;

// Query Parameters
export const queryReceiptsSchema = z.object({
  page: z.string().optional().transform(val => val ? parseInt(val) : 1),
  limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
  client_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  payment_method: z.enum(['Cash', 'Cheque', 'Bank Transfer', 'Mobile Money', 'Card', 'Other']).optional(),
  status: z.enum(['Pending', 'Cleared', 'Bounced', 'Cancelled']).optional(),
  start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
});

export type QueryReceiptsDto = z.infer<typeof queryReceiptsSchema>;

// ================================================
// Collection Activity DTOs
// ================================================

// Create Collection Activity
export const createCollectionActivitySchema = z.object({
  client_id: z.number().int().positive(),
  invoice_id: z.number().int().positive().optional(),
  activity_type: z.enum(['Email', 'Phone', 'Letter', 'Meeting', 'Legal Action', 'Other']),
  activity_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  contact_person: z.string().max(200).optional(),
  notes: z.string().max(5000),
  outcome: z.string().max(1000).optional(),
  follow_up_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
});

export type CreateCollectionActivityDto = z.infer<typeof createCollectionActivitySchema>;

// Update Collection Activity
export const updateCollectionActivitySchema = z.object({
  activity_type: z.enum(['Email', 'Phone', 'Letter', 'Meeting', 'Legal Action', 'Other']).optional(),
  activity_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  contact_person: z.string().max(200).optional(),
  notes: z.string().max(5000).optional(),
  outcome: z.string().max(1000).optional(),
  follow_up_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
});

export type UpdateCollectionActivityDto = z.infer<typeof updateCollectionActivitySchema>;

// Query Collection Activities
export const queryCollectionActivitiesSchema = z.object({
  page: z.string().optional().transform(val => val ? parseInt(val) : 1),
  limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
  client_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  invoice_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  activity_type: z.enum(['Email', 'Phone', 'Letter', 'Meeting', 'Legal Action', 'Other']).optional(),
  start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
});

export type QueryCollectionActivitiesDto = z.infer<typeof queryCollectionActivitiesSchema>;

// ================================================
// Response Types
// ================================================

export interface ReceiptResponse {
  id: number;
  receipt_number: string;
  client_id: number;
  client_name: string;
  receipt_date: string;
  amount: number;
  payment_method: 'Cash' | 'Cheque' | 'Bank Transfer' | 'Mobile Money' | 'Card' | 'Other';
  reference_number: string | null;
  bank_name: string | null;
  cheque_number: string | null;
  notes: string | null;
  status: 'Pending' | 'Cleared' | 'Bounced' | 'Cancelled';
  recorded_by: number;
  recorded_by_name: string;
  created_at: string;
  updated_at: string;
  allocations?: PaymentAllocationResponse[];
}

export interface PaymentAllocationResponse {
  id: number;
  receipt_id: number;
  invoice_id: number;
  invoice_number: string;
  amount: number;
  created_at: string;
}

export interface CollectionActivityResponse {
  id: number;
  client_id: number;
  client_name: string;
  invoice_id: number | null;
  invoice_number: string | null;
  activity_type: 'Email' | 'Phone' | 'Letter' | 'Meeting' | 'Legal Action' | 'Other';
  activity_date: string;
  contact_person: string | null;
  notes: string;
  outcome: string | null;
  follow_up_date: string | null;
  recorded_by: number;
  recorded_by_name: string;
  created_at: string;
  updated_at: string;
}

export interface ReceiptSummaryResponse {
  total_receipts: number;
  total_amount: number;
  pending_clearance: number;
  cleared_amount: number;
  bounced_count: number;
  by_payment_method: {
    payment_method: string;
    count: number;
    amount: number;
  }[];
}

export interface CollectionSummaryResponse {
  total_activities: number;
  activities_this_month: number;
  pending_follow_ups: number;
  by_activity_type: {
    activity_type: string;
    count: number;
  }[];
}
