import { RowDataPacket, ResultSetHeader } from 'mysql2/promise';
import { getPool } from '../../db/pool.js';
import type {
  CreateReceiptDto,
  UpdateReceiptDto,
  QueryReceiptsDto,
  CreateCollectionActivityDto,
  UpdateCollectionActivityDto,
  QueryCollectionActivitiesDto,
  ReceiptResponse,
  PaymentAllocationResponse,
  CollectionActivityResponse,
  ReceiptSummaryResponse,
  CollectionSummaryResponse,
} from './dto.js';

export class ReceiptRepository {
  // ================================================
  // Receipts
  // ================================================

  async listReceipts(query: QueryReceiptsDto) {
    const pool = await getPool();
    const { page = 1, limit = 50, client_id, payment_method, status, start_date, end_date } = query;
    const offset = (page - 1) * limit;

    let whereConditions: string[] = [];
    let params: any[] = [];

    if (client_id) {
      whereConditions.push('r.client_id = ?');
      params.push(client_id);
    }

    if (payment_method) {
      whereConditions.push('r.payment_method = ?');
      params.push(payment_method);
    }

    if (status) {
      whereConditions.push('r.status = ?');
      params.push(status);
    }

    if (start_date) {
      whereConditions.push('r.receipt_date >= ?');
      params.push(start_date);
    }

    if (end_date) {
      whereConditions.push('r.receipt_date <= ?');
      params.push(end_date);
    }

    const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';

    // Get total count
    const countQuery = `SELECT COUNT(*) as total FROM receipts r ${whereClause}`;
    const [countRows] = await pool.query<RowDataPacket[]>(countQuery, params);
    const total = countRows[0].total;

    // Get receipts
    const query_sql = `
      SELECT 
        r.*,
        c.name as client_name,
        u.username as recorded_by_name
      FROM receipts r
      JOIN clients c ON r.client_id = c.id
      JOIN users u ON r.recorded_by = u.id
      ${whereClause}
      ORDER BY r.receipt_date DESC, r.created_at DESC
      LIMIT ? OFFSET ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query_sql, [...params, limit, offset]);

    return {
      receipts: rows as ReceiptResponse[],
      pagination: {
        page,
        limit,
        total,
        totalPages: Math.ceil(total / limit),
      },
    };
  }

  async getReceiptById(id: number, includeAllocations: boolean = true): Promise<ReceiptResponse | null> {
    const pool = await getPool();

    const query = `
      SELECT 
        r.*,
        c.name as client_name,
        u.username as recorded_by_name
      FROM receipts r
      JOIN clients c ON r.client_id = c.id
      JOIN users u ON r.recorded_by = u.id
      WHERE r.id = ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, [id]);

    if (rows.length === 0) {
      return null;
    }

    const receipt = rows[0] as ReceiptResponse;

    if (includeAllocations) {
      receipt.allocations = await this.getReceiptAllocations(id);
    }

    return receipt;
  }

  async getReceiptAllocations(receiptId: number): Promise<PaymentAllocationResponse[]> {
    const pool = await getPool();

    const query = `
      SELECT 
        pa.*,
        i.invoice_number
      FROM payment_allocations pa
      JOIN invoices i ON pa.invoice_id = i.id
      WHERE pa.receipt_id = ?
      ORDER BY pa.id
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, [receiptId]);
    return rows as PaymentAllocationResponse[];
  }

  async generateReceiptNumber(year: number): Promise<string> {
    const pool = await getPool();

    // Get the last receipt number for this year
    const query = `
      SELECT receipt_number 
      FROM receipts 
      WHERE receipt_number LIKE ?
      ORDER BY receipt_number DESC 
      LIMIT 1
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, [`RCT-${year}-%`]);

    let nextNumber = 1;
    if (rows.length > 0) {
      const lastNumber = rows[0].receipt_number;
      const parts = lastNumber.split('-');
      if (parts.length === 3) {
        nextNumber = parseInt(parts[2]) + 1;
      }
    }

    return `RCT-${year}-${nextNumber.toString().padStart(5, '0')}`;
  }

  async createReceipt(data: CreateReceiptDto, userId: number): Promise<ReceiptResponse | null> {
    const pool = await getPool();
    const connection = await pool.getConnection();

    try {
      await connection.beginTransaction();

      // Validate total allocations equal receipt amount
      const totalAllocated = data.allocations.reduce((sum, alloc) => sum + alloc.amount, 0);
      if (Math.abs(totalAllocated - data.amount) > 0.01) {
        throw new Error('Total allocation amount must equal receipt amount');
      }

      // Validate all invoices belong to the client and have balance due
      for (const allocation of data.allocations) {
        const [invoices] = await connection.query<RowDataPacket[]>(
          'SELECT id, client_id, balance_due FROM invoices WHERE id = ?',
          [allocation.invoice_id]
        );

        if (invoices.length === 0) {
          throw new Error(`Invoice ${allocation.invoice_id} not found`);
        }

        const invoice = invoices[0];

        if (invoice.client_id !== data.client_id) {
          throw new Error(`Invoice ${allocation.invoice_id} does not belong to this client`);
        }

        if (allocation.amount > invoice.balance_due) {
          throw new Error(`Allocation amount exceeds balance due for invoice ${allocation.invoice_id}`);
        }
      }

      // Generate receipt number
      const receiptDate = new Date(data.receipt_date);
      const receiptNumber = await this.generateReceiptNumber(receiptDate.getFullYear());

      // Create receipt
      const receiptQuery = `
        INSERT INTO receipts (
          receipt_number, client_id, receipt_date, amount,
          payment_method, reference_number, bank_name, cheque_number,
          notes, recorded_by
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `;

      const [result] = await connection.query<ResultSetHeader>(receiptQuery, [
        receiptNumber,
        data.client_id,
        data.receipt_date,
        data.amount,
        data.payment_method,
        data.reference_number || null,
        data.bank_name || null,
        data.cheque_number || null,
        data.notes || null,
        userId,
      ]);

      const receiptId = result.insertId;

      // Create allocations and update invoices
      for (const allocation of data.allocations) {
        // Insert allocation
        await connection.query(
          `INSERT INTO payment_allocations (receipt_id, invoice_id, amount) VALUES (?, ?, ?)`,
          [receiptId, allocation.invoice_id, allocation.amount]
        );

        // Update invoice
        await connection.query(
          `UPDATE invoices 
           SET amount_paid = amount_paid + ?,
               balance_due = balance_due - ?,
               status = CASE 
                 WHEN balance_due - ? <= 0.01 THEN 'Paid'
                 WHEN amount_paid + ? > 0 THEN 'Partially Paid'
                 ELSE status
               END
           WHERE id = ?`,
          [allocation.amount, allocation.amount, allocation.amount, allocation.amount, allocation.invoice_id]
        );
      }

      await connection.commit();
      connection.release();

      return this.getReceiptById(receiptId);
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  }

  async updateReceipt(id: number, data: UpdateReceiptDto): Promise<ReceiptResponse | null> {
    const pool = await getPool();

    const existing = await this.getReceiptById(id, false);
    if (!existing) {
      throw new Error('Receipt not found');
    }

    // Only allow updates if status is Pending
    if (existing.status !== 'Pending' && !data.status) {
      throw new Error('Cannot update receipt that has been cleared or cancelled (except status)');
    }

    const updates: string[] = [];
    const params: any[] = [];

    if (data.receipt_date !== undefined) {
      updates.push('receipt_date = ?');
      params.push(data.receipt_date);
    }
    if (data.payment_method !== undefined) {
      updates.push('payment_method = ?');
      params.push(data.payment_method);
    }
    if (data.reference_number !== undefined) {
      updates.push('reference_number = ?');
      params.push(data.reference_number);
    }
    if (data.bank_name !== undefined) {
      updates.push('bank_name = ?');
      params.push(data.bank_name);
    }
    if (data.cheque_number !== undefined) {
      updates.push('cheque_number = ?');
      params.push(data.cheque_number);
    }
    if (data.notes !== undefined) {
      updates.push('notes = ?');
      params.push(data.notes);
    }
    if (data.status !== undefined) {
      updates.push('status = ?');
      params.push(data.status);
    }

    if (updates.length === 0) {
      return existing;
    }

    const query = `UPDATE receipts SET ${updates.join(', ')} WHERE id = ?`;
    await pool.query(query, [...params, id]);

    return this.getReceiptById(id);
  }

  async getReceiptSummary(clientId?: number): Promise<ReceiptSummaryResponse> {
    const pool = await getPool();

    let whereConditions: string[] = [];
    let params: any[] = [];

    if (clientId) {
      whereConditions.push('client_id = ?');
      params.push(clientId);
    }

    const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';

    const query = `
      SELECT 
        COUNT(*) as total_receipts,
        COALESCE(SUM(amount), 0) as total_amount,
        COALESCE(SUM(CASE WHEN status = 'Pending' THEN amount ELSE 0 END), 0) as pending_clearance,
        COALESCE(SUM(CASE WHEN status = 'Cleared' THEN amount ELSE 0 END), 0) as cleared_amount,
        SUM(CASE WHEN status = 'Bounced' THEN 1 ELSE 0 END) as bounced_count
      FROM receipts
      ${whereClause}
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, params);
    const summary = rows[0] as ReceiptSummaryResponse;

    // Get breakdown by payment method
    const methodQuery = `
      SELECT 
        payment_method,
        COUNT(*) as count,
        COALESCE(SUM(amount), 0) as amount
      FROM receipts
      ${whereClause}
      GROUP BY payment_method
      ORDER BY amount DESC
    `;

    const [methodRows] = await pool.query<RowDataPacket[]>(methodQuery, params);
    summary.by_payment_method = methodRows as any[];

    return summary;
  }

  // ================================================
  // Collection Activities
  // ================================================

  async listCollectionActivities(query: QueryCollectionActivitiesDto) {
    const pool = await getPool();
    const { page = 1, limit = 50, client_id, invoice_id, activity_type, start_date, end_date } = query;
    const offset = (page - 1) * limit;

    let whereConditions: string[] = [];
    let params: any[] = [];

    if (client_id) {
      whereConditions.push('ca.client_id = ?');
      params.push(client_id);
    }

    if (invoice_id) {
      whereConditions.push('ca.invoice_id = ?');
      params.push(invoice_id);
    }

    if (activity_type) {
      whereConditions.push('ca.activity_type = ?');
      params.push(activity_type);
    }

    if (start_date) {
      whereConditions.push('ca.activity_date >= ?');
      params.push(start_date);
    }

    if (end_date) {
      whereConditions.push('ca.activity_date <= ?');
      params.push(end_date);
    }

    const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';

    // Get total count
    const countQuery = `SELECT COUNT(*) as total FROM collection_activities ca ${whereClause}`;
    const [countRows] = await pool.query<RowDataPacket[]>(countQuery, params);
    const total = countRows[0].total;

    // Get activities
    const query_sql = `
      SELECT 
        ca.*,
        c.name as client_name,
        i.invoice_number,
        u.username as recorded_by_name
      FROM collection_activities ca
      JOIN clients c ON ca.client_id = c.id
      LEFT JOIN invoices i ON ca.invoice_id = i.id
      JOIN users u ON ca.recorded_by = u.id
      ${whereClause}
      ORDER BY ca.activity_date DESC, ca.created_at DESC
      LIMIT ? OFFSET ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query_sql, [...params, limit, offset]);

    return {
      activities: rows as CollectionActivityResponse[],
      pagination: {
        page,
        limit,
        total,
        totalPages: Math.ceil(total / limit),
      },
    };
  }

  async getCollectionActivityById(id: number): Promise<CollectionActivityResponse | null> {
    const pool = await getPool();

    const query = `
      SELECT 
        ca.*,
        c.name as client_name,
        i.invoice_number,
        u.username as recorded_by_name
      FROM collection_activities ca
      JOIN clients c ON ca.client_id = c.id
      LEFT JOIN invoices i ON ca.invoice_id = i.id
      JOIN users u ON ca.recorded_by = u.id
      WHERE ca.id = ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, [id]);
    return rows.length > 0 ? (rows[0] as CollectionActivityResponse) : null;
  }

  async createCollectionActivity(data: CreateCollectionActivityDto, userId: number): Promise<CollectionActivityResponse | null> {
    const pool = await getPool();

    const query = `
      INSERT INTO collection_activities (
        client_id, invoice_id, activity_type, activity_date,
        contact_person, notes, outcome, follow_up_date, recorded_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;

    const [result] = await pool.query<ResultSetHeader>(query, [
      data.client_id,
      data.invoice_id || null,
      data.activity_type,
      data.activity_date,
      data.contact_person || null,
      data.notes,
      data.outcome || null,
      data.follow_up_date || null,
      userId,
    ]);

    return this.getCollectionActivityById(result.insertId);
  }

  async updateCollectionActivity(id: number, data: UpdateCollectionActivityDto): Promise<CollectionActivityResponse | null> {
    const pool = await getPool();

    const existing = await this.getCollectionActivityById(id);
    if (!existing) {
      throw new Error('Collection activity not found');
    }

    const updates: string[] = [];
    const params: any[] = [];

    if (data.activity_type !== undefined) {
      updates.push('activity_type = ?');
      params.push(data.activity_type);
    }
    if (data.activity_date !== undefined) {
      updates.push('activity_date = ?');
      params.push(data.activity_date);
    }
    if (data.contact_person !== undefined) {
      updates.push('contact_person = ?');
      params.push(data.contact_person);
    }
    if (data.notes !== undefined) {
      updates.push('notes = ?');
      params.push(data.notes);
    }
    if (data.outcome !== undefined) {
      updates.push('outcome = ?');
      params.push(data.outcome);
    }
    if (data.follow_up_date !== undefined) {
      updates.push('follow_up_date = ?');
      params.push(data.follow_up_date);
    }

    if (updates.length === 0) {
      return existing;
    }

    const query = `UPDATE collection_activities SET ${updates.join(', ')} WHERE id = ?`;
    await pool.query(query, [...params, id]);

    return this.getCollectionActivityById(id);
  }

  async getCollectionSummary(clientId?: number): Promise<CollectionSummaryResponse> {
    const pool = await getPool();

    let whereConditions: string[] = [];
    let params: any[] = [];

    if (clientId) {
      whereConditions.push('client_id = ?');
      params.push(clientId);
    }

    const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';

    const query = `
      SELECT 
        COUNT(*) as total_activities,
        SUM(CASE WHEN activity_date >= DATE_SUB(CURDATE(), INTERVAL 1 MONTH) THEN 1 ELSE 0 END) as activities_this_month,
        SUM(CASE WHEN follow_up_date IS NOT NULL AND follow_up_date >= CURDATE() THEN 1 ELSE 0 END) as pending_follow_ups
      FROM collection_activities
      ${whereClause}
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, params);
    const summary = rows[0] as CollectionSummaryResponse;

    // Get breakdown by activity type
    const typeQuery = `
      SELECT 
        activity_type,
        COUNT(*) as count
      FROM collection_activities
      ${whereClause}
      GROUP BY activity_type
      ORDER BY count DESC
    `;

    const [typeRows] = await pool.query<RowDataPacket[]>(typeQuery, params);
    summary.by_activity_type = typeRows as any[];

    return summary;
  }
}

export const receiptRepo = new ReceiptRepository();
