import { Router, Request, Response, NextFunction } from 'express';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { receiptRepo } from './repo.js';
import {
  createReceiptSchema,
  updateReceiptSchema,
  queryReceiptsSchema,
  createCollectionActivitySchema,
  updateCollectionActivitySchema,
  queryCollectionActivitiesSchema,
} from './dto.js';

const router = Router();

// ================================================
// Receipts
// ================================================

/**
 * GET /api/receipts
 * List receipts with filters
 */
router.get(
  '/',
  authenticate,
  requirePermission('receipt:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const query = queryReceiptsSchema.parse(req.query);
      const result = await receiptRepo.listReceipts(query);
      res.json({ success: true, data: result });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/receipts/summary
 * Get receipt summary statistics
 */
router.get(
  '/summary',
  authenticate,
  requirePermission('receipt:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const clientId = req.query.client_id ? parseInt(req.query.client_id as string) : undefined;
      const summary = await receiptRepo.getReceiptSummary(clientId);
      res.json({ success: true, data: summary });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/receipts/:id
 * Get single receipt by ID (with allocations)
 */
router.get(
  '/:id',
  authenticate,
  requirePermission('receipt:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const receipt = await receiptRepo.getReceiptById(id, true);

      if (!receipt) {
        res.status(404).json({ success: false, error: 'Receipt not found' });
        return;
      }

      res.json({ success: true, data: receipt });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/receipts
 * Create new receipt with payment allocations
 */
router.post(
  '/',
  authenticate,
  requirePermission('receipt:create'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const data = createReceiptSchema.parse(req.body);
      const receipt = await receiptRepo.createReceipt(data, req.user!.id);
      res.status(201).json({ success: true, data: receipt });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * PUT /api/receipts/:id
 * Update receipt (Pending receipts only, or status changes)
 */
router.put(
  '/:id',
  authenticate,
  requirePermission('receipt:update'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const data = updateReceiptSchema.parse(req.body);
      const receipt = await receiptRepo.updateReceipt(id, data);
      res.json({ success: true, data: receipt });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/receipts/:id/allocations
 * Get payment allocations for a receipt
 */
router.get(
  '/:id/allocations',
  authenticate,
  requirePermission('receipt:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const allocations = await receiptRepo.getReceiptAllocations(id);
      res.json({ success: true, data: { allocations } });
    } catch (error) {
      next(error);
    }
  }
);

// ================================================
// Collection Activities
// ================================================

/**
 * GET /api/receipts/collections/activities
 * List collection activities with filters
 */
router.get(
  '/collections/activities',
  authenticate,
  requirePermission('receipt:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const query = queryCollectionActivitiesSchema.parse(req.query);
      const result = await receiptRepo.listCollectionActivities(query);
      res.json({ success: true, data: result });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/receipts/collections/summary
 * Get collection activities summary
 */
router.get(
  '/collections/summary',
  authenticate,
  requirePermission('receipt:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const clientId = req.query.client_id ? parseInt(req.query.client_id as string) : undefined;
      const summary = await receiptRepo.getCollectionSummary(clientId);
      res.json({ success: true, data: summary });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/receipts/collections/activities/:id
 * Get single collection activity
 */
router.get(
  '/collections/activities/:id',
  authenticate,
  requirePermission('receipt:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const activity = await receiptRepo.getCollectionActivityById(id);

      if (!activity) {
        res.status(404).json({ success: false, error: 'Collection activity not found' });
        return;
      }

      res.json({ success: true, data: activity });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/receipts/collections/activities
 * Create new collection activity
 */
router.post(
  '/collections/activities',
  authenticate,
  requirePermission('receipt:create'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const data = createCollectionActivitySchema.parse(req.body);
      const activity = await receiptRepo.createCollectionActivity(data, req.user!.id);
      res.status(201).json({ success: true, data: activity });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * PUT /api/receipts/collections/activities/:id
 * Update collection activity
 */
router.put(
  '/collections/activities/:id',
  authenticate,
  requirePermission('receipt:update'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const data = updateCollectionActivitySchema.parse(req.body);
      const activity = await receiptRepo.updateCollectionActivity(id, data);
      res.json({ success: true, data: activity });
    } catch (error) {
      next(error);
    }
  }
);

export default router;
