import { getPool } from '../../db/pool.js';
import { CreateRoleDto, UpdateRoleDto, RoleResponse } from './dto.js';

export class RoleRepository {
  /**
   * Get all roles with department info
   */
  async findAll(departmentId?: number): Promise<RoleResponse[]> {
    const pool = await getPool();
    
    let query = `
      SELECT r.id, r.name, r.department_id, d.name as department_name, r.description, r.created_at
      FROM roles r
      JOIN departments d ON d.id = r.department_id
    `;
    
    const params: any[] = [];
    
    if (departmentId) {
      query += ' WHERE r.department_id = ?';
      params.push(departmentId);
    }
    
    query += ' ORDER BY d.name ASC, r.name ASC';
    
    const [rows] = await pool.query<any[]>(query, params);

    return rows;
  }

  /**
   * Get role by ID
   */
  async findById(id: number): Promise<RoleResponse | null> {
    const pool = await getPool();
    
    const [rows] = await pool.query<any[]>(
      `SELECT r.id, r.name, r.department_id, d.name as department_name, r.description, r.created_at
       FROM roles r
       JOIN departments d ON d.id = r.department_id
       WHERE r.id = ?`,
      [id]
    );

    return rows.length > 0 ? rows[0] : null;
  }

  /**
   * Check if role name exists in department (excluding specific ID for updates)
   */
  async nameExistsInDepartment(
    name: string,
    departmentId: number,
    excludeId?: number
  ): Promise<boolean> {
    const pool = await getPool();
    
    const query = excludeId
      ? 'SELECT COUNT(*) as count FROM roles WHERE name = ? AND department_id = ? AND id != ?'
      : 'SELECT COUNT(*) as count FROM roles WHERE name = ? AND department_id = ?';
    
    const params = excludeId ? [name, departmentId, excludeId] : [name, departmentId];
    
    const [rows] = await pool.query<any[]>(query, params);
    
    return rows[0].count > 0;
  }

  /**
   * Check if department exists
   */
  async departmentExists(departmentId: number): Promise<boolean> {
    const pool = await getPool();
    
    const [rows] = await pool.query<any[]>(
      'SELECT COUNT(*) as count FROM departments WHERE id = ?',
      [departmentId]
    );

    return rows[0].count > 0;
  }

  /**
   * Create role
   */
  async create(data: CreateRoleDto): Promise<RoleResponse> {
    const pool = await getPool();
    
    const [result] = await pool.query<any>(
      `INSERT INTO roles (name, department_id, description)
       VALUES (?, ?, ?)`,
      [data.name, data.department_id, data.description || null]
    );

    const newRole = await this.findById(result.insertId);
    if (!newRole) {
      throw new Error('Failed to create role');
    }

    return newRole;
  }

  /**
   * Update role
   */
  async update(id: number, data: UpdateRoleDto): Promise<RoleResponse> {
    const pool = await getPool();
    
    const updates: string[] = [];
    const values: any[] = [];

    if (data.name !== undefined) {
      updates.push('name = ?');
      values.push(data.name);
    }

    if (data.department_id !== undefined) {
      updates.push('department_id = ?');
      values.push(data.department_id);
    }

    if (data.description !== undefined) {
      updates.push('description = ?');
      values.push(data.description);
    }

    if (updates.length === 0) {
      const role = await this.findById(id);
      if (!role) {
        throw new Error('Role not found');
      }
      return role;
    }

    values.push(id);

    await pool.query(
      `UPDATE roles SET ${updates.join(', ')} WHERE id = ?`,
      values
    );

    const updated = await this.findById(id);
    if (!updated) {
      throw new Error('Role not found');
    }

    return updated;
  }

  /**
   * Delete role
   */
  async delete(id: number): Promise<void> {
    const pool = await getPool();
    
    await pool.query('DELETE FROM roles WHERE id = ?', [id]);
  }

  /**
   * Check if role has associated users
   */
  async hasUsers(id: number): Promise<boolean> {
    const pool = await getPool();
    
    const [rows] = await pool.query<any[]>(
      'SELECT COUNT(*) as count FROM users WHERE role_id = ?',
      [id]
    );

    return rows[0].count > 0;
  }

  /**
   * Log role action to audit log
   */
  async logAction(
    userId: number,
    roleId: number,
    action: 'CREATE' | 'UPDATE' | 'DELETE',
    meta: Record<string, any>
  ): Promise<void> {
    const pool = await getPool();
    
    await pool.query(
      `INSERT INTO audit_logs (user_id, entity, entity_id, action, meta)
       VALUES (?, 'role', ?, ?, ?)`,
      [userId, roleId, action, JSON.stringify(meta)]
    );
  }
}
