import { Router, Request, Response, NextFunction } from 'express';
import { RoleRepository } from './repo.js';
import {
  createRoleSchema,
  updateRoleSchema,
  deleteRoleSchema,
  getRoleSchema,
} from './dto.js';
import { validate } from '../../middleware/validate.js';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { AppError } from '../../middleware/errors.js';

const router = Router();
const roleRepo = new RoleRepository();

// All routes require authentication
router.use(authenticate);

/**
 * GET /api/roles
 * List all roles (optionally filtered by department_id query param)
 */
router.get(
  '/',
  async (req: Request, res: Response, next: NextFunction): Promise<void> => {
    try {
      const departmentId = req.query.department_id ? parseInt(req.query.department_id as string) : undefined;
      
      let roles = await roleRepo.findAll();
      
      // Filter by department if specified
      if (departmentId) {
        roles = roles.filter(r => r.department_id === departmentId);
      }

      res.json({
        success: true,
        data: { roles },
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/roles/:id
 * Get role by ID
 */
router.get(
  '/:id',
  validate(getRoleSchema),
  async (req: Request, res: Response, next: NextFunction): Promise<void> => {
    try {
      const id = parseInt(req.params.id);
      const role = await roleRepo.findById(id);

      if (!role) {
        throw new AppError(404, 'Role not found');
      }

      res.json({
        success: true,
        data: { role },
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/roles
 * Create new role (Admin/Manager only)
 */
router.post(
  '/',
  requirePermission('roles:create'),
  validate(createRoleSchema),
  async (req: Request, res: Response, next: NextFunction): Promise<void> => {
    try {
      const { name, department_id, description } = req.body;

      // Check if department exists
      const deptExists = await roleRepo.departmentExists(department_id);
      if (!deptExists) {
        throw new AppError(404, 'Department not found');
      }

      // Check for duplicate name in department
      const exists = await roleRepo.nameExistsInDepartment(name, department_id);
      if (exists) {
        throw new AppError(409, 'Role name already exists in this department');
      }

      const role = await roleRepo.create({ name, department_id, description });

      // Log action
      await roleRepo.logAction(req.user!.id, role.id, 'CREATE', {
        name: role.name,
        department_id: role.department_id,
        description: role.description,
      });

      res.status(201).json({
        success: true,
        data: { role },
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * PUT /api/roles/:id
 * Update role (Admin/Manager only)
 */
router.put(
  '/:id',
  requirePermission('roles:update'),
  validate(updateRoleSchema),
  async (req: Request, res: Response, next: NextFunction): Promise<void> => {
    try {
      const id = parseInt(req.params.id);
      const { name, department_id, description } = req.body;

      // Check if role exists
      const existing = await roleRepo.findById(id);
      if (!existing) {
        throw new AppError(404, 'Role not found');
      }

      // If department is changing, check if new department exists
      if (department_id && department_id !== existing.department_id) {
        const deptExists = await roleRepo.departmentExists(department_id);
        if (!deptExists) {
          throw new AppError(404, 'Department not found');
        }
      }

      // Check for duplicate name in department
      const targetDeptId = department_id || existing.department_id;
      if (name && (name !== existing.name || targetDeptId !== existing.department_id)) {
        const exists = await roleRepo.nameExistsInDepartment(name, targetDeptId, id);
        if (exists) {
          throw new AppError(409, 'Role name already exists in this department');
        }
      }

      const role = await roleRepo.update(id, { name, department_id, description });

      // Log action
      await roleRepo.logAction(req.user!.id, id, 'UPDATE', {
        old: existing,
        new: role,
      });

      res.json({
        success: true,
        data: { role },
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * DELETE /api/roles/:id
 * Delete role (Admin only)
 */
router.delete(
  '/:id',
  requirePermission('roles:delete'),
  validate(deleteRoleSchema),
  async (req: Request, res: Response, next: NextFunction): Promise<void> => {
    try {
      const id = parseInt(req.params.id);

      // Check if role exists
      const role = await roleRepo.findById(id);
      if (!role) {
        throw new AppError(404, 'Role not found');
      }

      // Check if role has users
      const hasUsers = await roleRepo.hasUsers(id);
      if (hasUsers) {
        throw new AppError(
          400,
          'Cannot delete role with associated users. Reassign users first.'
        );
      }

      await roleRepo.delete(id);

      // Log action
      await roleRepo.logAction(req.user!.id, id, 'DELETE', {
        name: role.name,
        department_id: role.department_id,
      });

      res.json({
        success: true,
        data: { message: 'Role deleted successfully' },
      });
    } catch (error) {
      next(error);
    }
  }
);

export default router;
