import { z } from 'zod';

// ================================================
// Time Entry DTOs
// ================================================

// Create Time Entry
export const createTimeEntrySchema = z.object({
  matter_id: z.number().int().positive(),
  task_id: z.number().int().positive().optional(),
  entry_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  hours: z.number().positive().max(24),
  activity_code: z.string().min(1).max(50),
  description: z.string().min(1).max(5000),
  billable: z.boolean().default(true),
  hourly_rate: z.number().positive().optional(),
});

export type CreateTimeEntryDto = z.infer<typeof createTimeEntrySchema>;

// Update Time Entry
export const updateTimeEntrySchema = z.object({
  entry_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  hours: z.number().positive().max(24).optional(),
  activity_code: z.string().min(1).max(50).optional(),
  description: z.string().min(1).max(5000).optional(),
  billable: z.boolean().optional(),
  hourly_rate: z.number().positive().optional(),
});

export type UpdateTimeEntryDto = z.infer<typeof updateTimeEntrySchema>;

// Query Parameters
export const queryTimeEntriesSchema = z.object({
  page: z.string().optional().transform(val => val ? parseInt(val) : 1),
  limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
  user_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  matter_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  status: z.enum(['Draft', 'Submitted', 'Approved', 'Rejected', 'Invoiced']).optional(),
  start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  billable: z.string().optional().transform(val => val === 'true' ? true : val === 'false' ? false : undefined),
});

export type QueryTimeEntriesDto = z.infer<typeof queryTimeEntriesSchema>;

// Submit for Approval
export const submitTimeEntrySchema = z.object({
  entry_ids: z.array(z.number().int().positive()).min(1),
});

export type SubmitTimeEntryDto = z.infer<typeof submitTimeEntrySchema>;

// Approve Time Entry
export const approveTimeEntrySchema = z.object({
  entry_ids: z.array(z.number().int().positive()).min(1),
});

export type ApproveTimeEntryDto = z.infer<typeof approveTimeEntrySchema>;

// Reject Time Entry
export const rejectTimeEntrySchema = z.object({
  entry_ids: z.array(z.number().int().positive()).min(1),
  rejection_reason: z.string().min(1).max(1000),
});

export type RejectTimeEntryDto = z.infer<typeof rejectTimeEntrySchema>;

// ================================================
// Timer DTOs
// ================================================

// Start Timer
export const startTimerSchema = z.object({
  matter_id: z.number().int().positive(),
  activity_code: z.string().min(1).max(50),
  description: z.string().max(5000).optional(),
});

export type StartTimerDto = z.infer<typeof startTimerSchema>;

// Stop Timer (no body needed, uses user context)

// ================================================
// Activity Code DTOs
// ================================================

// Create Activity Code (admin only)
export const createActivityCodeSchema = z.object({
  code: z.string().min(1).max(50),
  name: z.string().min(1).max(100),
  description: z.string().max(500).optional(),
  default_billable: z.boolean().default(true),
  sort_order: z.number().int().default(0),
});

export type CreateActivityCodeDto = z.infer<typeof createActivityCodeSchema>;

// Update Activity Code
export const updateActivityCodeSchema = z.object({
  name: z.string().min(1).max(100).optional(),
  description: z.string().max(500).optional(),
  default_billable: z.boolean().optional(),
  is_active: z.boolean().optional(),
  sort_order: z.number().int().optional(),
});

export type UpdateActivityCodeDto = z.infer<typeof updateActivityCodeSchema>;

// ================================================
// Response Types
// ================================================

export interface TimeEntryResponse {
  id: number;
  user_id: number;
  user_name: string;
  matter_id: number;
  matter_number: string | null;
  matter_title: string;
  task_id: number | null;
  entry_date: string;
  start_time: string | null;
  end_time: string | null;
  hours: number;
  activity_code: string;
  activity_name: string;
  description: string;
  billable: boolean;
  hourly_rate: number | null;
  total_amount: number | null;
  status: 'Draft' | 'Submitted' | 'Approved' | 'Rejected' | 'Invoiced';
  submitted_at: string | null;
  approved_by: number | null;
  approved_by_name: string | null;
  approved_at: string | null;
  rejection_reason: string | null;
  created_at: string;
  updated_at: string;
}

export interface RunningTimerResponse {
  id: number;
  user_id: number;
  matter_id: number;
  matter_number: string | null;
  matter_title: string;
  activity_code: string;
  activity_name: string;
  description: string | null;
  start_time: string;
  elapsed_seconds: number;
  elapsed_formatted: string;
}

export interface ActivityCodeResponse {
  id: number;
  code: string;
  name: string;
  description: string | null;
  default_billable: boolean;
  is_active: boolean;
  sort_order: number;
  created_at: string;
  updated_at: string;
}
