import { Router, Request, Response, NextFunction } from 'express';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { timeRepo } from './repo.js';
import {
  createTimeEntrySchema,
  updateTimeEntrySchema,
  queryTimeEntriesSchema,
  submitTimeEntrySchema,
  approveTimeEntrySchema,
  rejectTimeEntrySchema,
  startTimerSchema,
  createActivityCodeSchema,
  updateActivityCodeSchema,
} from './dto.js';

const router = Router();

// ================================================
// Time Entries
// ================================================

/**
 * GET /api/time
 * List time entries with filters
 */
router.get(
  '/',
  authenticate,
  requirePermission('time:create'), // All users with time:create can view
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const query = queryTimeEntriesSchema.parse(req.query);
      // Check if user has view-all permission via role
      const canViewAll = false; // Will implement role-based check later
      const result = await timeRepo.listTimeEntries(query, req.user!.id, canViewAll);
      res.json({ success: true, data: result });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/time/:id
 * Get single time entry by ID
 */
router.get(
  '/:id',
  authenticate,
  requirePermission('time:create'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const canViewAll = false; // Will implement role-based check later
      const entry = await timeRepo.getTimeEntryById(id, req.user!.id, canViewAll);

      if (!entry) {
        res.status(404).json({ success: false, error: 'Time entry not found' });
        return;
      }

      res.json({ success: true, data: entry });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/time
 * Create new time entry
 */
router.post(
  '/',
  authenticate,
  requirePermission('time:create'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const data = createTimeEntrySchema.parse(req.body);
      const entry = await timeRepo.createTimeEntry(data, req.user!.id);
      res.status(201).json({ success: true, data: entry });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * PUT /api/time/:id
 * Update time entry (only Draft or Rejected entries)
 */
router.put(
  '/:id',
  authenticate,
  requirePermission('time:create'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const data = updateTimeEntrySchema.parse(req.body);
      const canViewAll = false; // Will implement role-based check later
      const entry = await timeRepo.updateTimeEntry(id, data, req.user!.id, canViewAll);
      res.json({ success: true, data: entry });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * DELETE /api/time/:id
 * Delete time entry (only Draft entries)
 */
router.delete(
  '/:id',
  authenticate,
  requirePermission('time:create'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const canViewAll = false; // Will implement role-based check later
      await timeRepo.deleteTimeEntry(id, req.user!.id, canViewAll);
      res.json({ success: true, message: 'Time entry deleted successfully' });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/time/submit
 * Submit time entries for approval
 */
router.post(
  '/submit',
  authenticate,
  requirePermission('time:create'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { entry_ids } = submitTimeEntrySchema.parse(req.body);
      await timeRepo.submitTimeEntries(entry_ids, req.user!.id);
      res.json({ success: true, message: `${entry_ids.length} time entries submitted for approval` });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/time/approve
 * Approve time entries (managers/admins only)
 */
router.post(
  '/approve',
  authenticate,
  requirePermission('time:approve'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { entry_ids } = approveTimeEntrySchema.parse(req.body);
      await timeRepo.approveTimeEntries(entry_ids, req.user!.id);
      res.json({ success: true, message: `${entry_ids.length} time entries approved` });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/time/reject
 * Reject time entries (managers/admins only)
 */
router.post(
  '/reject',
  authenticate,
  requirePermission('time:approve'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { entry_ids, rejection_reason } = rejectTimeEntrySchema.parse(req.body);
      await timeRepo.rejectTimeEntries(entry_ids, req.user!.id, rejection_reason);
      res.json({ success: true, message: `${entry_ids.length} time entries rejected` });
    } catch (error) {
      next(error);
    }
  }
);

// ================================================
// Timers
// ================================================

/**
 * GET /api/time/timer/current
 * Get current running timer for user
 */
router.get(
  '/timer/current',
  authenticate,
  requirePermission('time:create'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const timer = await timeRepo.getRunningTimer(req.user!.id);
      if (!timer) {
        res.json({ success: true, data: null });
        return;
      }
      res.json({ success: true, data: timer });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/time/timer/start
 * Start a new timer
 */
router.post(
  '/timer/start',
  authenticate,
  requirePermission('time:create'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const data = startTimerSchema.parse(req.body);
      const timer = await timeRepo.startTimer(data, req.user!.id);
      res.status(201).json({ success: true, data: timer });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/time/timer/stop
 * Stop current timer and create time entry
 */
router.post(
  '/timer/stop',
  authenticate,
  requirePermission('time:create'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const entry = await timeRepo.stopTimer(req.user!.id);
      res.json({ success: true, data: entry, message: 'Timer stopped and time entry created' });
    } catch (error) {
      next(error);
    }
  }
);

// ================================================
// Activity Codes
// ================================================

/**
 * GET /api/time/activity-codes
 * List activity codes
 */
router.get(
  '/activity-codes',
  authenticate,
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const activeOnly = req.query.active_only !== 'false';
      const codes = await timeRepo.listActivityCodes(activeOnly);
      res.json({ success: true, data: { codes } });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/time/activity-codes/:id
 * Get single activity code
 */
router.get(
  '/activity-codes/:id',
  authenticate,
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const code = await timeRepo.getActivityCodeById(id);

      if (!code) {
        res.status(404).json({ success: false, error: 'Activity code not found' });
        return;
      }

      res.json({ success: true, data: code });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/time/activity-codes
 * Create activity code (admin only)
 */
router.post(
  '/activity-codes',
  authenticate,
  requirePermission('time:approve'), // Only managers/admins
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const data = createActivityCodeSchema.parse(req.body);
      const code = await timeRepo.createActivityCode(data);
      res.status(201).json({ success: true, data: code });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * PUT /api/time/activity-codes/:id
 * Update activity code (admin only)
 */
router.put(
  '/activity-codes/:id',
  authenticate,
  requirePermission('time:approve'), // Only managers/admins
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const data = updateActivityCodeSchema.parse(req.body);
      const code = await timeRepo.updateActivityCode(id, data);
      res.json({ success: true, data: code });
    } catch (error) {
      next(error);
    }
  }
);

export default router;
