import { z } from 'zod';

// ================================================
// Trust Account DTOs
// ================================================

// Create Trust Account
export const createTrustAccountSchema = z.object({
  client_id: z.number().int().positive(),
  matter_id: z.number().int().positive().optional(),
  account_name: z.string().min(1).max(200),
  account_number: z.string().max(50).optional(),
  bank_name: z.string().max(100).optional(),
  currency: z.string().length(3).default('GHS'),
  account_type: z.enum(['Client Trust', 'Matter Trust', 'General Trust']).default('Client Trust'),
  notes: z.string().max(2000).optional(),
});

export type CreateTrustAccountDto = z.infer<typeof createTrustAccountSchema>;

// Update Trust Account
export const updateTrustAccountSchema = z.object({
  account_name: z.string().min(1).max(200).optional(),
  account_number: z.string().max(50).optional(),
  bank_name: z.string().max(100).optional(),
  notes: z.string().max(2000).optional(),
  is_active: z.boolean().optional(),
});

export type UpdateTrustAccountDto = z.infer<typeof updateTrustAccountSchema>;

// ================================================
// Trust Transaction DTOs
// ================================================

// Create Trust Transaction (Deposit/Withdrawal)
export const createTrustTransactionSchema = z.object({
  trust_account_id: z.number().int().positive(),
  transaction_type: z.enum(['Deposit', 'Withdrawal', 'Transfer Out', 'Transfer In', 'Fee Transfer', 'Refund']),
  amount: z.number().positive(),
  transaction_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  reference_number: z.string().max(100).optional(),
  description: z.string().min(1).max(5000),
  payment_method: z.string().max(50).optional(),
  payee: z.string().max(200).optional(),
  related_invoice_id: z.number().int().positive().optional(),
  related_matter_id: z.number().int().positive().optional(),
  requires_approval: z.boolean().default(true),
});

export type CreateTrustTransactionDto = z.infer<typeof createTrustTransactionSchema>;

// Transfer Between Trust Accounts
export const transferBetweenTrustSchema = z.object({
  from_account_id: z.number().int().positive(),
  to_account_id: z.number().int().positive(),
  amount: z.number().positive(),
  transaction_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  description: z.string().min(1).max(5000),
  reference_number: z.string().max(100).optional(),
});

export type TransferBetweenTrustDto = z.infer<typeof transferBetweenTrustSchema>;

// Approve Trust Transaction
export const approveTrustTransactionSchema = z.object({
  approval_notes: z.string().max(1000).optional(),
});

export type ApproveTrustTransactionDto = z.infer<typeof approveTrustTransactionSchema>;

// Reject Trust Transaction
export const rejectTrustTransactionSchema = z.object({
  rejection_reason: z.string().min(1).max(1000),
});

export type RejectTrustTransactionDto = z.infer<typeof rejectTrustTransactionSchema>;

// Query Trust Transactions
export const queryTrustTransactionsSchema = z.object({
  page: z.string().optional().transform(val => val ? parseInt(val) : 1),
  limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
  trust_account_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  transaction_type: z.enum(['Deposit', 'Withdrawal', 'Transfer Out', 'Transfer In', 'Fee Transfer', 'Refund']).optional(),
  status: z.enum(['Pending', 'Approved', 'Rejected', 'Posted']).optional(),
  start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
  end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
});

export type QueryTrustTransactionsDto = z.infer<typeof queryTrustTransactionsSchema>;

// ================================================
// Trust Reconciliation DTOs
// ================================================

// Create Trust Reconciliation
export const createTrustReconciliationSchema = z.object({
  trust_account_id: z.number().int().positive(),
  reconciliation_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
  bank_statement_balance: z.number(),
  book_balance: z.number(),
  reconciled_balance: z.number(),
  unreconciled_items: z.string().max(5000).optional(),
  notes: z.string().max(2000).optional(),
});

export type CreateTrustReconciliationDto = z.infer<typeof createTrustReconciliationSchema>;

// Update Trust Reconciliation
export const updateTrustReconciliationSchema = z.object({
  bank_statement_balance: z.number().optional(),
  reconciled_balance: z.number().optional(),
  unreconciled_items: z.string().max(5000).optional(),
  notes: z.string().max(2000).optional(),
  status: z.enum(['Draft', 'Completed', 'Reviewed']).optional(),
});

export type UpdateTrustReconciliationDto = z.infer<typeof updateTrustReconciliationSchema>;

// ================================================
// Response Types
// ================================================

export interface TrustAccountResponse {
  id: number;
  client_id: number;
  client_name: string;
  matter_id: number | null;
  matter_number: string | null;
  matter_title: string | null;
  account_name: string;
  account_number: string | null;
  bank_name: string | null;
  currency: string;
  account_type: 'Client Trust' | 'Matter Trust' | 'General Trust';
  current_balance: number;
  notes: string | null;
  is_active: boolean;
  created_by: number;
  created_by_name: string;
  created_at: string;
  updated_at: string;
}

export interface TrustTransactionResponse {
  id: number;
  trust_account_id: number;
  trust_account_name: string;
  transaction_type: 'Deposit' | 'Withdrawal' | 'Transfer Out' | 'Transfer In' | 'Fee Transfer' | 'Refund';
  amount: number;
  balance_after: number;
  transaction_date: string;
  reference_number: string | null;
  description: string;
  payment_method: string | null;
  payee: string | null;
  related_invoice_id: number | null;
  related_matter_id: number | null;
  status: 'Pending' | 'Approved' | 'Rejected' | 'Posted';
  requires_approval: boolean;
  created_by: number;
  created_by_name: string;
  approved_by: number | null;
  approved_by_name: string | null;
  approved_at: string | null;
  approval_notes: string | null;
  rejection_reason: string | null;
  posted_at: string | null;
  is_reversed: boolean;
  reversal_of_id: number | null;
  created_at: string;
  updated_at: string;
}

export interface TrustReconciliationResponse {
  id: number;
  trust_account_id: number;
  trust_account_name: string;
  reconciliation_date: string;
  bank_statement_balance: number;
  book_balance: number;
  reconciled_balance: number;
  difference: number;
  unreconciled_items: string | null;
  notes: string | null;
  status: 'Draft' | 'Completed' | 'Reviewed';
  reconciled_by: number;
  reconciled_by_name: string;
  reviewed_by: number | null;
  reviewed_by_name: string | null;
  reviewed_at: string | null;
  created_at: string;
  updated_at: string;
}

export interface TrustSummaryResponse {
  total_accounts: number;
  active_accounts: number;
  total_balance: number;
  pending_approvals: number;
  pending_amount: number;
  last_reconciliation_date: string | null;
  by_account_type: {
    account_type: string;
    count: number;
    total_balance: number;
  }[];
}
