import { RowDataPacket, ResultSetHeader } from 'mysql2/promise';
import { getPool } from '../../db/pool.js';
import type {
  CreateTrustAccountDto,
  UpdateTrustAccountDto,
  CreateTrustTransactionDto,
  TransferBetweenTrustDto,
  ApproveTrustTransactionDto,
  RejectTrustTransactionDto,
  QueryTrustTransactionsDto,
  CreateTrustReconciliationDto,
  UpdateTrustReconciliationDto,
  TrustAccountResponse,
  TrustTransactionResponse,
  TrustReconciliationResponse,
  TrustSummaryResponse,
} from './dto.js';

export class TrustRepository {
  // ================================================
  // Trust Accounts
  // ================================================

  async listTrustAccounts(clientId?: number, isActive?: boolean) {
    const pool = await getPool();

    let whereConditions: string[] = [];
    let params: any[] = [];

    if (clientId) {
      whereConditions.push('ta.client_id = ?');
      params.push(clientId);
    }

    if (isActive !== undefined) {
      whereConditions.push('ta.is_active = ?');
      params.push(isActive);
    }

    const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';

    const query = `
      SELECT 
        ta.*,
        c.name as client_name,
        m.matter_number,
        m.title as matter_title,
        u.username as created_by_name
      FROM trust_accounts ta
      JOIN clients c ON ta.client_id = c.id
      LEFT JOIN matters m ON ta.matter_id = m.id
      JOIN users u ON ta.created_by = u.id
      ${whereClause}
      ORDER BY ta.created_at DESC
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, params);
    return rows as TrustAccountResponse[];
  }

  async getTrustAccountById(id: number): Promise<TrustAccountResponse | null> {
    const pool = await getPool();

    const query = `
      SELECT 
        ta.*,
        c.name as client_name,
        m.matter_number,
        m.title as matter_title,
        u.username as created_by_name
      FROM trust_accounts ta
      JOIN clients c ON ta.client_id = c.id
      LEFT JOIN matters m ON ta.matter_id = m.id
      JOIN users u ON ta.created_by = u.id
      WHERE ta.id = ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, [id]);
    return rows.length > 0 ? (rows[0] as TrustAccountResponse) : null;
  }

  async createTrustAccount(data: CreateTrustAccountDto, userId: number): Promise<TrustAccountResponse | null> {
    const pool = await getPool();

    const query = `
      INSERT INTO trust_accounts (
        client_id, matter_id, account_name, account_number,
        bank_name, currency, account_type, notes, created_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;

    const [result] = await pool.query<ResultSetHeader>(query, [
      data.client_id,
      data.matter_id || null,
      data.account_name,
      data.account_number || null,
      data.bank_name || null,
      data.currency,
      data.account_type,
      data.notes || null,
      userId,
    ]);

    return this.getTrustAccountById(result.insertId);
  }

  async updateTrustAccount(id: number, data: UpdateTrustAccountDto): Promise<TrustAccountResponse | null> {
    const pool = await getPool();

    const existing = await this.getTrustAccountById(id);
    if (!existing) {
      throw new Error('Trust account not found');
    }

    const updates: string[] = [];
    const params: any[] = [];

    if (data.account_name !== undefined) {
      updates.push('account_name = ?');
      params.push(data.account_name);
    }
    if (data.account_number !== undefined) {
      updates.push('account_number = ?');
      params.push(data.account_number);
    }
    if (data.bank_name !== undefined) {
      updates.push('bank_name = ?');
      params.push(data.bank_name);
    }
    if (data.notes !== undefined) {
      updates.push('notes = ?');
      params.push(data.notes);
    }
    if (data.is_active !== undefined) {
      updates.push('is_active = ?');
      params.push(data.is_active);
    }

    if (updates.length === 0) {
      return existing;
    }

    const query = `UPDATE trust_accounts SET ${updates.join(', ')} WHERE id = ?`;
    await pool.query(query, [...params, id]);

    return this.getTrustAccountById(id);
  }

  // ================================================
  // Trust Transactions (Immutable Ledger)
  // ================================================

  async listTrustTransactions(query: QueryTrustTransactionsDto) {
    const pool = await getPool();
    const { page = 1, limit = 50, trust_account_id, transaction_type, status, start_date, end_date } = query;
    const offset = (page - 1) * limit;

    let whereConditions: string[] = [];
    let params: any[] = [];

    if (trust_account_id) {
      whereConditions.push('tt.trust_account_id = ?');
      params.push(trust_account_id);
    }

    if (transaction_type) {
      whereConditions.push('tt.transaction_type = ?');
      params.push(transaction_type);
    }

    if (status) {
      whereConditions.push('tt.status = ?');
      params.push(status);
    }

    if (start_date) {
      whereConditions.push('tt.transaction_date >= ?');
      params.push(start_date);
    }

    if (end_date) {
      whereConditions.push('tt.transaction_date <= ?');
      params.push(end_date);
    }

    const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';

    // Get total count
    const countQuery = `SELECT COUNT(*) as total FROM trust_transactions tt ${whereClause}`;
    const [countRows] = await pool.query<RowDataPacket[]>(countQuery, params);
    const total = countRows[0].total;

    // Get transactions
    const query_sql = `
      SELECT 
        tt.*,
        ta.account_name as trust_account_name,
        creator.username as created_by_name,
        approver.username as approved_by_name
      FROM trust_transactions tt
      JOIN trust_accounts ta ON tt.trust_account_id = ta.id
      JOIN users creator ON tt.created_by = creator.id
      LEFT JOIN users approver ON tt.approved_by = approver.id
      ${whereClause}
      ORDER BY tt.transaction_date DESC, tt.created_at DESC
      LIMIT ? OFFSET ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query_sql, [...params, limit, offset]);

    return {
      transactions: rows as TrustTransactionResponse[],
      pagination: {
        page,
        limit,
        total,
        totalPages: Math.ceil(total / limit),
      },
    };
  }

  async getTrustTransactionById(id: number): Promise<TrustTransactionResponse | null> {
    const pool = await getPool();

    const query = `
      SELECT 
        tt.*,
        ta.account_name as trust_account_name,
        creator.username as created_by_name,
        approver.username as approved_by_name
      FROM trust_transactions tt
      JOIN trust_accounts ta ON tt.trust_account_id = ta.id
      JOIN users creator ON tt.created_by = creator.id
      LEFT JOIN users approver ON tt.approved_by = approver.id
      WHERE tt.id = ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, [id]);
    return rows.length > 0 ? (rows[0] as TrustTransactionResponse) : null;
  }

  async createTrustTransaction(data: CreateTrustTransactionDto, userId: number): Promise<TrustTransactionResponse | null> {
    const pool = await getPool();
    const connection = await pool.getConnection();

    try {
      await connection.beginTransaction();

      // Get current balance
      const account = await this.getTrustAccountById(data.trust_account_id);
      if (!account) {
        throw new Error('Trust account not found');
      }

      // Calculate new balance
      let balance_after = account.current_balance;
      if (data.transaction_type === 'Deposit' || data.transaction_type === 'Transfer In') {
        balance_after += data.amount;
      } else if (data.transaction_type === 'Withdrawal' || data.transaction_type === 'Transfer Out' || data.transaction_type === 'Fee Transfer' || data.transaction_type === 'Refund') {
        balance_after -= data.amount;
      }

      // Check for negative balance
      if (balance_after < 0) {
        throw new Error('Insufficient trust account balance');
      }

      // Determine initial status
      const initialStatus = data.requires_approval ? 'Pending' : 'Posted';

      // Create transaction (immutable)
      const query = `
        INSERT INTO trust_transactions (
          trust_account_id, transaction_type, amount, balance_after,
          transaction_date, reference_number, description, payment_method,
          payee, related_invoice_id, related_matter_id,
          status, requires_approval, created_by
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `;

      const [result] = await connection.query<ResultSetHeader>(query, [
        data.trust_account_id,
        data.transaction_type,
        data.amount,
        balance_after,
        data.transaction_date,
        data.reference_number || null,
        data.description,
        data.payment_method || null,
        data.payee || null,
        data.related_invoice_id || null,
        data.related_matter_id || null,
        initialStatus,
        data.requires_approval,
        userId,
      ]);

      // Update account balance if not requiring approval
      if (!data.requires_approval) {
        await connection.query(
          'UPDATE trust_accounts SET current_balance = ? WHERE id = ?',
          [balance_after, data.trust_account_id]
        );
      }

      await connection.commit();
      connection.release();

      return this.getTrustTransactionById(result.insertId);
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  }

  async transferBetweenTrust(data: TransferBetweenTrustDto, userId: number): Promise<{ from: TrustTransactionResponse | null; to: TrustTransactionResponse | null }> {
    const pool = await getPool();
    const connection = await pool.getConnection();

    try {
      await connection.beginTransaction();

      // Create withdrawal from source account
      const withdrawalData: CreateTrustTransactionDto = {
        trust_account_id: data.from_account_id,
        transaction_type: 'Transfer Out',
        amount: data.amount,
        transaction_date: data.transaction_date,
        reference_number: data.reference_number,
        description: `Transfer to trust account: ${data.description}`,
        requires_approval: true, // Always require approval for transfers
      };

      const fromTransaction = await this.createTrustTransaction(withdrawalData, userId);

      // Create deposit to destination account
      const depositData: CreateTrustTransactionDto = {
        trust_account_id: data.to_account_id,
        transaction_type: 'Transfer In',
        amount: data.amount,
        transaction_date: data.transaction_date,
        reference_number: data.reference_number,
        description: `Transfer from trust account: ${data.description}`,
        requires_approval: true, // Always require approval for transfers
      };

      const toTransaction = await this.createTrustTransaction(depositData, userId);

      await connection.commit();
      connection.release();

      return { from: fromTransaction, to: toTransaction };
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  }

  async approveTrustTransaction(id: number, userId: number, data: ApproveTrustTransactionDto): Promise<TrustTransactionResponse | null> {
    const pool = await getPool();
    const connection = await pool.getConnection();

    try {
      await connection.beginTransaction();

      const transaction = await this.getTrustTransactionById(id);
      if (!transaction) {
        throw new Error('Trust transaction not found');
      }

      if (transaction.status !== 'Pending') {
        throw new Error('Only pending transactions can be approved');
      }

      // Dual control: Approver cannot be the creator
      if (transaction.created_by === userId) {
        throw new Error('You cannot approve your own transaction (dual control violation)');
      }

      // Update transaction status
      await connection.query(
        `UPDATE trust_transactions 
         SET status = 'Approved', approved_by = ?, approved_at = NOW(), approval_notes = ?
         WHERE id = ?`,
        [userId, data.approval_notes || null, id]
      );

      // Update trust account balance
      await connection.query(
        'UPDATE trust_accounts SET current_balance = ? WHERE id = ?',
        [transaction.balance_after, transaction.trust_account_id]
      );

      // Mark as posted
      await connection.query(
        'UPDATE trust_transactions SET status = "Posted", posted_at = NOW() WHERE id = ?',
        [id]
      );

      await connection.commit();
      connection.release();

      return this.getTrustTransactionById(id);
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  }

  async rejectTrustTransaction(id: number, userId: number, data: RejectTrustTransactionDto): Promise<TrustTransactionResponse | null> {
    const pool = await getPool();

    const transaction = await this.getTrustTransactionById(id);
    if (!transaction) {
      throw new Error('Trust transaction not found');
    }

    if (transaction.status !== 'Pending') {
      throw new Error('Only pending transactions can be rejected');
    }

    // Dual control: Approver cannot be the creator
    if (transaction.created_by === userId) {
      throw new Error('You cannot reject your own transaction (dual control violation)');
    }

    await pool.query(
      `UPDATE trust_transactions 
       SET status = 'Rejected', approved_by = ?, approved_at = NOW(), rejection_reason = ?
       WHERE id = ?`,
      [userId, data.rejection_reason, id]
    );

    return this.getTrustTransactionById(id);
  }

  // ================================================
  // Trust Reconciliations
  // ================================================

  async listTrustReconciliations(trustAccountId?: number) {
    const pool = await getPool();

    let whereConditions: string[] = [];
    let params: any[] = [];

    if (trustAccountId) {
      whereConditions.push('tr.trust_account_id = ?');
      params.push(trustAccountId);
    }

    const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';

    const query = `
      SELECT 
        tr.*,
        ta.account_name as trust_account_name,
        reconciler.username as reconciled_by_name,
        reviewer.username as reviewed_by_name
      FROM trust_reconciliations tr
      JOIN trust_accounts ta ON tr.trust_account_id = ta.id
      JOIN users reconciler ON tr.reconciled_by = reconciler.id
      LEFT JOIN users reviewer ON tr.reviewed_by = reviewer.id
      ${whereClause}
      ORDER BY tr.reconciliation_date DESC
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, params);
    return rows as TrustReconciliationResponse[];
  }

  async getTrustReconciliationById(id: number): Promise<TrustReconciliationResponse | null> {
    const pool = await getPool();

    const query = `
      SELECT 
        tr.*,
        ta.account_name as trust_account_name,
        reconciler.username as reconciled_by_name,
        reviewer.username as reviewed_by_name
      FROM trust_reconciliations tr
      JOIN trust_accounts ta ON tr.trust_account_id = ta.id
      JOIN users reconciler ON tr.reconciled_by = reconciler.id
      LEFT JOIN users reviewer ON tr.reviewed_by = reviewer.id
      WHERE tr.id = ?
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query, [id]);
    return rows.length > 0 ? (rows[0] as TrustReconciliationResponse) : null;
  }

  async createTrustReconciliation(data: CreateTrustReconciliationDto, userId: number): Promise<TrustReconciliationResponse | null> {
    const pool = await getPool();

    const difference = data.bank_statement_balance - data.reconciled_balance;

    const query = `
      INSERT INTO trust_reconciliations (
        trust_account_id, reconciliation_date,
        bank_statement_balance, book_balance, reconciled_balance,
        difference, unreconciled_items, notes, reconciled_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;

    const [result] = await pool.query<ResultSetHeader>(query, [
      data.trust_account_id,
      data.reconciliation_date,
      data.bank_statement_balance,
      data.book_balance,
      data.reconciled_balance,
      difference,
      data.unreconciled_items || null,
      data.notes || null,
      userId,
    ]);

    return this.getTrustReconciliationById(result.insertId);
  }

  async updateTrustReconciliation(id: number, data: UpdateTrustReconciliationDto): Promise<TrustReconciliationResponse | null> {
    const pool = await getPool();

    const existing = await this.getTrustReconciliationById(id);
    if (!existing) {
      throw new Error('Trust reconciliation not found');
    }

    const updates: string[] = [];
    const params: any[] = [];

    if (data.bank_statement_balance !== undefined) {
      updates.push('bank_statement_balance = ?');
      params.push(data.bank_statement_balance);
    }
    if (data.reconciled_balance !== undefined) {
      updates.push('reconciled_balance = ?');
      params.push(data.reconciled_balance);
      
      // Recalculate difference
      const bankBalance = data.bank_statement_balance !== undefined ? data.bank_statement_balance : existing.bank_statement_balance;
      updates.push('difference = ?');
      params.push(bankBalance - data.reconciled_balance);
    }
    if (data.unreconciled_items !== undefined) {
      updates.push('unreconciled_items = ?');
      params.push(data.unreconciled_items);
    }
    if (data.notes !== undefined) {
      updates.push('notes = ?');
      params.push(data.notes);
    }
    if (data.status !== undefined) {
      updates.push('status = ?');
      params.push(data.status);
    }

    if (updates.length === 0) {
      return existing;
    }

    const query = `UPDATE trust_reconciliations SET ${updates.join(', ')} WHERE id = ?`;
    await pool.query(query, [...params, id]);

    return this.getTrustReconciliationById(id);
  }

  async reviewTrustReconciliation(id: number, userId: number): Promise<TrustReconciliationResponse | null> {
    const pool = await getPool();

    const reconciliation = await this.getTrustReconciliationById(id);
    if (!reconciliation) {
      throw new Error('Trust reconciliation not found');
    }

    if (reconciliation.reconciled_by === userId) {
      throw new Error('You cannot review your own reconciliation (dual control violation)');
    }

    await pool.query(
      'UPDATE trust_reconciliations SET status = "Reviewed", reviewed_by = ?, reviewed_at = NOW() WHERE id = ?',
      [userId, id]
    );

    return this.getTrustReconciliationById(id);
  }

  async getTrustSummary(): Promise<TrustSummaryResponse> {
    const pool = await getPool();

    const query = `
      SELECT 
        COUNT(*) as total_accounts,
        SUM(CASE WHEN is_active = TRUE THEN 1 ELSE 0 END) as active_accounts,
        COALESCE(SUM(CASE WHEN is_active = TRUE THEN current_balance ELSE 0 END), 0) as total_balance
      FROM trust_accounts
    `;

    const [rows] = await pool.query<RowDataPacket[]>(query);
    const summary = rows[0] as TrustSummaryResponse;

    // Get pending approvals
    const pendingQuery = `
      SELECT 
        COUNT(*) as pending_approvals,
        COALESCE(SUM(ABS(amount)), 0) as pending_amount
      FROM trust_transactions
      WHERE status = 'Pending'
    `;

    const [pendingRows] = await pool.query<RowDataPacket[]>(pendingQuery);
    summary.pending_approvals = pendingRows[0].pending_approvals;
    summary.pending_amount = pendingRows[0].pending_amount;

    // Get last reconciliation date
    const reconQuery = `
      SELECT MAX(reconciliation_date) as last_reconciliation_date
      FROM trust_reconciliations
    `;

    const [reconRows] = await pool.query<RowDataPacket[]>(reconQuery);
    summary.last_reconciliation_date = reconRows[0].last_reconciliation_date;

    // Get breakdown by account type
    const typeQuery = `
      SELECT 
        account_type,
        COUNT(*) as count,
        COALESCE(SUM(current_balance), 0) as total_balance
      FROM trust_accounts
      WHERE is_active = TRUE
      GROUP BY account_type
      ORDER BY total_balance DESC
    `;

    const [typeRows] = await pool.query<RowDataPacket[]>(typeQuery);
    summary.by_account_type = typeRows as any[];

    return summary;
  }
}

export const trustRepo = new TrustRepository();
