import { getPool } from '../../db/pool.js';
import { hashPassword } from '../../utils/crypto.js';
import { CreateUserDto, UpdateUserDto, UserResponse } from './dto.js';

export class UserRepository {
  /**
   * Get all users with employee, role, and department info
   */
  async findAll(): Promise<UserResponse[]> {
    const pool = await getPool();
    
    const [rows] = await pool.query<any[]>(
      `SELECT 
        u.id, u.username, u.employee_id,
        CONCAT(e.first_name, ' ', e.last_name) as employee_name,
        u.role_id, r.name as role_name,
        e.department_id, d.name as department_name,
        u.created_at
       FROM users u
       JOIN employees e ON e.id = u.employee_id
       JOIN roles r ON r.id = u.role_id
       JOIN departments d ON d.id = e.department_id
       ORDER BY u.username ASC`
    );

    return rows;
  }

  /**
   * Get user by ID
   */
  async findById(id: number): Promise<UserResponse | null> {
    const pool = await getPool();
    
    const [rows] = await pool.query<any[]>(
      `SELECT 
        u.id, u.username, u.employee_id,
        CONCAT(e.first_name, ' ', e.last_name) as employee_name,
        u.role_id, r.name as role_name,
        e.department_id, d.name as department_name,
        u.created_at
       FROM users u
       JOIN employees e ON e.id = u.employee_id
       JOIN roles r ON r.id = u.role_id
       JOIN departments d ON d.id = e.department_id
       WHERE u.id = ?`,
      [id]
    );

    return rows.length > 0 ? rows[0] : null;
  }

  /**
   * Check if username exists (excluding specific ID for updates)
   */
  async usernameExists(username: string, excludeId?: number): Promise<boolean> {
    const pool = await getPool();
    
    const query = excludeId
      ? 'SELECT COUNT(*) as count FROM users WHERE username = ? AND id != ?'
      : 'SELECT COUNT(*) as count FROM users WHERE username = ?';
    
    const params = excludeId ? [username, excludeId] : [username];
    
    const [rows] = await pool.query<any[]>(query, params);
    
    return rows[0].count > 0;
  }

  /**
   * Check if employee already has a user account
   */
  async employeeHasUser(employeeId: number, excludeUserId?: number): Promise<boolean> {
    const pool = await getPool();
    
    const query = excludeUserId
      ? 'SELECT COUNT(*) as count FROM users WHERE employee_id = ? AND id != ?'
      : 'SELECT COUNT(*) as count FROM users WHERE employee_id = ?';
    
    const params = excludeUserId ? [employeeId, excludeUserId] : [employeeId];
    
    const [rows] = await pool.query<any[]>(query, params);
    
    return rows[0].count > 0;
  }

  /**
   * Check if employee exists
   */
  async employeeExists(employeeId: number): Promise<boolean> {
    const pool = await getPool();
    
    const [rows] = await pool.query<any[]>(
      'SELECT COUNT(*) as count FROM employees WHERE id = ?',
      [employeeId]
    );

    return rows[0].count > 0;
  }

  /**
   * Check if role exists
   */
  async roleExists(roleId: number): Promise<boolean> {
    const pool = await getPool();
    
    const [rows] = await pool.query<any[]>(
      'SELECT COUNT(*) as count FROM roles WHERE id = ?',
      [roleId]
    );

    return rows[0].count > 0;
  }

  /**
   * Create user
   */
  async create(data: CreateUserDto): Promise<UserResponse> {
    const pool = await getPool();
    
    const passwordHash = await hashPassword(data.password);
    
    const [result] = await pool.query<any>(
      `INSERT INTO users (username, password_hash, employee_id, role_id)
       VALUES (?, ?, ?, ?)`,
      [data.username, passwordHash, data.employee_id, data.role_id]
    );

    const newUser = await this.findById(result.insertId);
    if (!newUser) {
      throw new Error('Failed to create user');
    }

    return newUser;
  }

  /**
   * Update user
   */
  async update(id: number, data: UpdateUserDto): Promise<UserResponse> {
    const pool = await getPool();
    
    const updates: string[] = [];
    const values: any[] = [];

    if (data.username !== undefined) {
      updates.push('username = ?');
      values.push(data.username);
    }

    if (data.password !== undefined) {
      const passwordHash = await hashPassword(data.password);
      updates.push('password_hash = ?');
      values.push(passwordHash);
    }

    if (data.employee_id !== undefined) {
      updates.push('employee_id = ?');
      values.push(data.employee_id);
    }

    if (data.role_id !== undefined) {
      updates.push('role_id = ?');
      values.push(data.role_id);
    }

    if (updates.length === 0) {
      const user = await this.findById(id);
      if (!user) {
        throw new Error('User not found');
      }
      return user;
    }

    values.push(id);

    await pool.query(
      `UPDATE users SET ${updates.join(', ')} WHERE id = ?`,
      values
    );

    const updated = await this.findById(id);
    if (!updated) {
      throw new Error('User not found');
    }

    return updated;
  }

  /**
   * Delete user
   */
  async delete(id: number): Promise<void> {
    const pool = await getPool();
    
    // Delete associated sessions first
    await pool.query('DELETE FROM sessions WHERE user_id = ?', [id]);
    
    // Delete user
    await pool.query('DELETE FROM users WHERE id = ?', [id]);
  }

  /**
   * Log user action to audit log
   */
  async logAction(
    userId: number,
    targetUserId: number,
    action: 'CREATE' | 'UPDATE' | 'DELETE',
    meta: Record<string, any>
  ): Promise<void> {
    const pool = await getPool();
    
    await pool.query(
      `INSERT INTO audit_logs (user_id, entity, entity_id, action, meta)
       VALUES (?, 'user', ?, ?, ?)`,
      [userId, targetUserId, action, JSON.stringify(meta)]
    );
  }
}
