import { z } from 'zod';

// ================================================
// WIP Summary DTOs
// ================================================

// Query WIP Summary
export const queryWIPSummarySchema = z.object({
  matter_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  client_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  min_balance: z.string().optional().transform(val => val ? parseFloat(val) : undefined),
});

export type QueryWIPSummaryDto = z.infer<typeof queryWIPSummarySchema>;

// ================================================
// Revenue Recognition DTOs
// ================================================

// Query Revenue Recognition
export const queryRevenueSchema = z.object({
  page: z.string().optional().transform(val => val ? parseInt(val) : 1),
  limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
  matter_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  client_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
  start_month: z.string().regex(/^\d{4}-\d{2}$/).optional(),
  end_month: z.string().regex(/^\d{4}-\d{2}$/).optional(),
});

export type QueryRevenueDto = z.infer<typeof queryRevenueSchema>;

// Create Revenue Recognition Entry
export const createRevenueSchema = z.object({
  matter_id: z.number().int().positive(),
  recognition_month: z.string().regex(/^\d{4}-\d{2}$/),
  time_value: z.number().min(0).default(0),
  expense_value: z.number().min(0).default(0),
  total_value: z.number().min(0),
  invoiced_amount: z.number().min(0).default(0),
  collected_amount: z.number().min(0).default(0),
  notes: z.string().max(2000).optional(),
});

export type CreateRevenueDto = z.infer<typeof createRevenueSchema>;

// Update Revenue Recognition Entry
export const updateRevenueSchema = z.object({
  time_value: z.number().min(0).optional(),
  expense_value: z.number().min(0).optional(),
  total_value: z.number().min(0).optional(),
  invoiced_amount: z.number().min(0).optional(),
  collected_amount: z.number().min(0).optional(),
  notes: z.string().max(2000).optional(),
});

export type UpdateRevenueDto = z.infer<typeof updateRevenueSchema>;

// ================================================
// Response Types
// ================================================

export interface WIPSummaryResponse {
  id: number;
  matter_id: number;
  matter_number: string | null;
  matter_title: string;
  client_id: number;
  client_name: string;
  unbilled_time_hours: number;
  unbilled_time_value: number;
  unbilled_expenses: number;
  total_wip: number;
  last_invoice_date: string | null;
  oldest_unbilled_date: string | null;
  last_updated: string;
}

export interface RevenueRecognitionResponse {
  id: number;
  matter_id: number;
  matter_number: string | null;
  matter_title: string;
  client_id: number;
  client_name: string;
  recognition_month: string;
  time_value: number;
  expense_value: number;
  total_value: number;
  invoiced_amount: number;
  collected_amount: number;
  unbilled_value: number;
  uncollected_value: number;
  notes: string | null;
  created_by: number;
  created_by_name: string;
  created_at: string;
  updated_at: string;
}

export interface WIPAnalyticsResponse {
  total_wip_value: number;
  total_unbilled_time: number;
  total_unbilled_expenses: number;
  avg_days_unbilled: number;
  matters_with_wip: number;
  top_wip_matters: {
    matter_id: number;
    matter_number: string | null;
    matter_title: string;
    client_name: string;
    wip_value: number;
  }[];
  wip_by_client: {
    client_id: number;
    client_name: string;
    wip_value: number;
    matter_count: number;
  }[];
}

export interface RevenueAnalyticsResponse {
  total_recognized: number;
  total_invoiced: number;
  total_collected: number;
  total_unbilled: number;
  total_uncollected: number;
  collection_rate: number;
  realization_rate: number;
  by_month: {
    month: string;
    recognized: number;
    invoiced: number;
    collected: number;
  }[];
}
