import { Router, Request, Response, NextFunction } from 'express';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { wipRepo } from './repo.js';
import {
  queryWIPSummarySchema,
  queryRevenueSchema,
  createRevenueSchema,
  updateRevenueSchema,
} from './dto.js';

const router = Router();

// ================================================
// WIP Summary
// ================================================

/**
 * GET /api/wip/summary
 * Get WIP summary with optional filters
 */
router.get(
  '/summary',
  authenticate,
  requirePermission('wip:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const query = queryWIPSummarySchema.parse(req.query);
      const summary = await wipRepo.listWIPSummary(query);
      res.json({ success: true, data: { summary } });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/wip/refresh
 * Manually refresh WIP summary
 */
router.post(
  '/refresh',
  authenticate,
  requirePermission('wip:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const matterId = req.body.matter_id ? parseInt(req.body.matter_id) : undefined;
      await wipRepo.refreshWIPSummary(matterId);
      res.json({ 
        success: true, 
        message: matterId ? `WIP refreshed for matter ${matterId}` : 'WIP refreshed for all matters' 
      });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/wip/analytics
 * Get WIP analytics and insights
 */
router.get(
  '/analytics',
  authenticate,
  requirePermission('wip:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const analytics = await wipRepo.getWIPAnalytics();
      res.json({ success: true, data: analytics });
    } catch (error) {
      next(error);
    }
  }
);

// ================================================
// Revenue Recognition
// ================================================

/**
 * GET /api/wip/revenue
 * List revenue recognition records
 */
router.get(
  '/revenue',
  authenticate,
  requirePermission('wip:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const query = queryRevenueSchema.parse(req.query);
      const result = await wipRepo.listRevenueRecognition(query);
      res.json({ success: true, data: result });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/wip/revenue/:id
 * Get single revenue recognition record
 */
router.get(
  '/revenue/:id',
  authenticate,
  requirePermission('wip:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const revenue = await wipRepo.getRevenueRecognitionById(id);

      if (!revenue) {
        res.status(404).json({ success: false, error: 'Revenue recognition record not found' });
        return;
      }

      res.json({ success: true, data: revenue });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/wip/revenue
 * Create revenue recognition record
 */
router.post(
  '/revenue',
  authenticate,
  requirePermission('wip:manage'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const data = createRevenueSchema.parse(req.body);
      const revenue = await wipRepo.createRevenueRecognition(data, req.user!.id);
      res.status(201).json({ success: true, data: revenue });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * PUT /api/wip/revenue/:id
 * Update revenue recognition record
 */
router.put(
  '/revenue/:id',
  authenticate,
  requirePermission('wip:manage'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const id = parseInt(req.params.id);
      const data = updateRevenueSchema.parse(req.body);
      const revenue = await wipRepo.updateRevenueRecognition(id, data);
      res.json({ success: true, data: revenue });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * GET /api/wip/revenue/analytics
 * Get revenue analytics
 */
router.get(
  '/revenue/analytics',
  authenticate,
  requirePermission('wip:read'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const startMonth = req.query.start_month as string | undefined;
      const endMonth = req.query.end_month as string | undefined;
      const analytics = await wipRepo.getRevenueAnalytics(startMonth, endMonth);
      res.json({ success: true, data: analytics });
    } catch (error) {
      next(error);
    }
  }
);

/**
 * POST /api/wip/revenue/auto-generate
 * Auto-generate revenue recognition for a month
 */
router.post(
  '/revenue/auto-generate',
  authenticate,
  requirePermission('wip:manage'),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      const month = req.body.month;
      if (!month || !/^\d{4}-\d{2}$/.test(month)) {
        res.status(400).json({ success: false, error: 'Invalid month format. Use YYYY-MM' });
        return;
      }

      const count = await wipRepo.autoGenerateRevenueForMonth(month, req.user!.id);
      res.json({ 
        success: true, 
        message: `Generated revenue recognition for ${count} matters`,
        data: { count, month }
      });
    } catch (error) {
      next(error);
    }
  }
);

export default router;
