import { z } from 'zod';

// ====================
// Request Schemas
// ====================

export const createWorkflowSchema = z.object({
  body: z.object({
    workflow_name: z.string().min(3).max(100),
    description: z.string().optional(),
    stages: z.array(z.object({
      name: z.string().min(2).max(100),
      order: z.number().int().positive(),
      sla_hours: z.number().int().positive().nullable(),
    })).min(2, 'Workflow must have at least 2 stages'),
    is_default: z.boolean().default(false),
  }),
});

export const updateWorkflowSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
  body: z.object({
    workflow_name: z.string().min(3).max(100).optional(),
    description: z.string().optional(),
    stages: z.array(z.object({
      name: z.string().min(2).max(100),
      order: z.number().int().positive(),
      sla_hours: z.number().int().positive().nullable(),
    })).min(2).optional(),
    is_default: z.boolean().optional(),
    is_active: z.boolean().optional(),
  }),
});

export const getWorkflowSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
});

export const transitionStageSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
  body: z.object({
    stage: z.string().min(2).max(100),
    transition_reason: z.string().optional(),
    notes: z.string().optional(),
  }),
});

export const getMatterStagesSchema = z.object({
  params: z.object({
    id: z.string().regex(/^\d+$/).transform(Number),
  }),
});

// ====================
// TypeScript Types
// ====================

export type CreateWorkflowDto = z.infer<typeof createWorkflowSchema>['body'];
export type UpdateWorkflowDto = z.infer<typeof updateWorkflowSchema>['body'];
export type TransitionStageDto = z.infer<typeof transitionStageSchema>['body'];

export interface WorkflowStage {
  name: string;
  order: number;
  sla_hours: number | null;
}

export interface WorkflowResponse {
  id: number;
  workflow_name: string;
  description: string | null;
  stages: WorkflowStage[];
  is_default: boolean;
  is_active: boolean;
  created_by: number | null;
  created_by_name: string | null;
  created_at: string;
  updated_at: string;
}

export interface MatterStageResponse {
  id: number;
  matter_id: number;
  stage: string;
  stage_order: number;
  started_at: string;
  ended_at: string | null;
  sla_hours: number | null;
  sla_due_at: string | null;
  is_overdue: boolean;
  transition_reason: string | null;
  notes: string | null;
  transitioned_by: number | null;
  transitioned_by_name: string | null;
  created_at: string;
}
