# Test department & role APIs

Write-Host "======================================" -ForegroundColor Cyan
Write-Host "Testing Departments & Roles APIs" -ForegroundColor Cyan  
Write-Host "======================================" -ForegroundColor Cyan
Write-Host ""

# Login first
$loginBody = @{
    username = "admin"
    password = "password"
} | ConvertTo-Json

$session = $null

try {
    $response = Invoke-WebRequest -Uri "http://localhost:8080/api/auth/login" `
        -Method POST `
        -ContentType "application/json" `
        -Body $loginBody `
        -SessionVariable session

    Write-Host "[OK] Logged in as admin" -ForegroundColor Green
    Write-Host ""

    # Test 1: Get all departments
    Write-Host "Test 1: GET /api/departments" -ForegroundColor Yellow
    $depts = Invoke-WebRequest -Uri "http://localhost:8080/api/departments" `
        -Method GET `
        -WebSession $session

    $deptsData = $depts.Content | ConvertFrom-Json
    Write-Host "[OK] Found" $deptsData.data.departments.Count "departments" -ForegroundColor Green
    Write-Host ""

    # Test 2: Get all roles
    Write-Host "Test 2: GET /api/roles" -ForegroundColor Yellow
    $roles = Invoke-WebRequest -Uri "http://localhost:8080/api/roles" `
        -Method GET `
        -WebSession $session

    $rolesData = $roles.Content | ConvertFrom-Json
    Write-Host "[OK] Found" $rolesData.data.roles.Count "roles" -ForegroundColor Green
    Write-Host ""

    # Test 3: Create new department
    Write-Host "Test 3: POST /api/departments (create)" -ForegroundColor Yellow
    $newDept = @{
        name = "Test Department"
        description = "Created via API test"
    } | ConvertTo-Json

    $createResp = Invoke-WebRequest -Uri "http://localhost:8080/api/departments" `
        -Method POST `
        -ContentType "application/json" `
        -Body $newDept `
        -WebSession $session

    $createdDept = $createResp.Content | ConvertFrom-Json
    $newDeptId = $createdDept.data.department.id
    Write-Host "[OK] Created department ID:" $newDeptId -ForegroundColor Green
    Write-Host ""

    # Test 4: Create role for new department
    Write-Host "Test 4: POST /api/roles (create)" -ForegroundColor Yellow
    $newRole = @{
        name = "Test Role"
        department_id = $newDeptId
        description = "Created via API test"
    } | ConvertTo-Json

    $createRoleResp = Invoke-WebRequest -Uri "http://localhost:8080/api/roles" `
        -Method POST `
        -ContentType "application/json" `
        -Body $newRole `
        -WebSession $session

    $createdRole = $createRoleResp.Content | ConvertFrom-Json
    $newRoleId = $createdRole.data.role.id
    Write-Host "[OK] Created role ID:" $newRoleId -ForegroundColor Green
    Write-Host ""

    # Test 5: Update department
    Write-Host "Test 5: PUT /api/departments/:id (update)" -ForegroundColor Yellow
    $updateDept = @{
        name = "Test Department Updated"
    } | ConvertTo-Json

    $updateResp = Invoke-WebRequest -Uri "http://localhost:8080/api/departments/$newDeptId" `
        -Method PUT `
        -ContentType "application/json" `
        -Body $updateDept `
        -WebSession $session

    Write-Host "[OK] Updated department" -ForegroundColor Green
    Write-Host ""

    # Test 6: Delete role (must delete before department)
    Write-Host "Test 6: DELETE /api/roles/:id" -ForegroundColor Yellow
    Invoke-WebRequest -Uri "http://localhost:8080/api/roles/$newRoleId" `
        -Method DELETE `
        -WebSession $session | Out-Null

    Write-Host "[OK] Deleted role" -ForegroundColor Green
    Write-Host ""

    # Test 7: Delete department
    Write-Host "Test 7: DELETE /api/departments/:id" -ForegroundColor Yellow
    Invoke-WebRequest -Uri "http://localhost:8080/api/departments/$newDeptId" `
        -Method DELETE `
        -WebSession $session | Out-Null

    Write-Host "[OK] Deleted department" -ForegroundColor Green
    Write-Host ""

    Write-Host "======================================" -ForegroundColor Cyan
    Write-Host "All tests passed!" -ForegroundColor Green
    Write-Host "======================================" -ForegroundColor Cyan

} catch {
    Write-Host "[FAIL] Test failed:" $_.Exception.Message -ForegroundColor Red
    if ($_.Exception.Response) {
        $reader = [System.IO.StreamReader]::new($_.Exception.Response.GetResponseStream())
        $errorBody = $reader.ReadToEnd()
        Write-Host "Error response:" $errorBody -ForegroundColor Red
    }
}
