# Test employee & user APIs

Write-Host "======================================" -ForegroundColor Cyan
Write-Host "Testing Employees & Users APIs" -ForegroundColor Cyan  
Write-Host "======================================" -ForegroundColor Cyan
Write-Host ""

# Login first
$loginBody = @{
    username = "admin"
    password = "password"
} | ConvertTo-Json

$session = $null

try {
    $response = Invoke-WebRequest -Uri "http://localhost:8080/api/auth/login" `
        -Method POST `
        -ContentType "application/json" `
        -Body $loginBody `
        -SessionVariable session

    Write-Host "[OK] Logged in as admin" -ForegroundColor Green
    Write-Host ""

    # Test 1: Get all employees
    Write-Host "Test 1: GET /api/employees" -ForegroundColor Yellow
    $emps = Invoke-WebRequest -Uri "http://localhost:8080/api/employees" `
        -Method GET `
        -WebSession $session

    $empsData = $emps.Content | ConvertFrom-Json
    Write-Host "[OK] Found" $empsData.data.employees.Count "employees" -ForegroundColor Green
    Write-Host ""

    # Test 2: Create new employee
    Write-Host "Test 2: POST /api/employees (create)" -ForegroundColor Yellow
    $newEmp = @{
        first_name = "Test"
        last_name = "Employee"
        email = "test.employee@lawfirm.com"
        phone = "555-0100"
        department_id = 1
        role_id = 1
        hire_date = "2024-01-15"
        salary = 75000
    } | ConvertTo-Json

    $createResp = Invoke-WebRequest -Uri "http://localhost:8080/api/employees" `
        -Method POST `
        -ContentType "application/json" `
        -Body $newEmp `
        -WebSession $session

    $createdEmp = $createResp.Content | ConvertFrom-Json
    $newEmpId = $createdEmp.data.employee.id
    Write-Host "[OK] Created employee ID:" $newEmpId -ForegroundColor Green
    Write-Host ""

    # Test 3: Get all users
    Write-Host "Test 3: GET /api/users" -ForegroundColor Yellow
    $users = Invoke-WebRequest -Uri "http://localhost:8080/api/users" `
        -Method GET `
        -WebSession $session

    $usersData = $users.Content | ConvertFrom-Json
    Write-Host "[OK] Found" $usersData.data.users.Count "users" -ForegroundColor Green
    Write-Host ""

    # Test 4: Create user for new employee
    Write-Host "Test 4: POST /api/users (create)" -ForegroundColor Yellow
    $newUser = @{
        username = "testuser"
        password = "password123"
        employee_id = $newEmpId
        role_id = 4
    } | ConvertTo-Json

    $createUserResp = Invoke-WebRequest -Uri "http://localhost:8080/api/users" `
        -Method POST `
        -ContentType "application/json" `
        -Body $newUser `
        -WebSession $session

    $createdUser = $createUserResp.Content | ConvertFrom-Json
    $newUserId = $createdUser.data.user.id
    Write-Host "[OK] Created user ID:" $newUserId -ForegroundColor Green
    Write-Host ""

    # Test 5: Update employee
    Write-Host "Test 5: PUT /api/employees/:id (update)" -ForegroundColor Yellow
    $updateEmp = @{
        salary = 80000
    } | ConvertTo-Json

    Invoke-WebRequest -Uri "http://localhost:8080/api/employees/$newEmpId" `
        -Method PUT `
        -ContentType "application/json" `
        -Body $updateEmp `
        -WebSession $session | Out-Null

    Write-Host "[OK] Updated employee" -ForegroundColor Green
    Write-Host ""

    # Test 6: Delete user (must delete before employee)
    Write-Host "Test 6: DELETE /api/users/:id" -ForegroundColor Yellow
    Invoke-WebRequest -Uri "http://localhost:8080/api/users/$newUserId" `
        -Method DELETE `
        -WebSession $session | Out-Null

    Write-Host "[OK] Deleted user" -ForegroundColor Green
    Write-Host ""

    # Test 7: Delete employee
    Write-Host "Test 7: DELETE /api/employees/:id" -ForegroundColor Yellow
    Invoke-WebRequest -Uri "http://localhost:8080/api/employees/$newEmpId" `
        -Method DELETE `
        -WebSession $session | Out-Null

    Write-Host "[OK] Deleted employee" -ForegroundColor Green
    Write-Host ""

    Write-Host "======================================" -ForegroundColor Cyan
    Write-Host "All tests passed!" -ForegroundColor Green
    Write-Host "======================================" -ForegroundColor Cyan

} catch {
    Write-Host "[FAIL] Test failed:" $_.Exception.Message -ForegroundColor Red
    if ($_.Exception.Response) {
        $reader = [System.IO.StreamReader]::new($_.Exception.Response.GetResponseStream())
        $errorBody = $reader.ReadToEnd()
        Write-Host "Error response:" $errorBody -ForegroundColor Red
    }
}
