import { getPool } from './pool.js';
import { hashPassword } from '../utils/crypto.js';
import { logger } from '../utils/logger.js';
async function seed() {
    const pool = await getPool();
    try {
        logger.info('Starting database seeding...');
        // ====================
        // Seed Departments
        // ====================
        logger.info('Seeding departments...');
        const [deptResult] = await pool.query(`INSERT INTO departments (name, description) VALUES
        ('Litigation', 'Handles all litigation and dispute resolution cases'),
        ('Corporate', 'Corporate law, compliance, and business advisory'),
        ('Family Law', 'Family matters, divorce, and custody cases')
      ON DUPLICATE KEY UPDATE id=id`);
        logger.info('✓ Departments seeded');
        // Get department IDs
        const [depts] = await pool.query('SELECT id, name FROM departments ORDER BY id');
        const deptMap = new Map(depts.map(d => [d.name, d.id]));
        // ====================
        // Seed Roles
        // ====================
        logger.info('Seeding roles...');
        await pool.query(`INSERT INTO roles (department_id, name, description) VALUES
        (${deptMap.get('Litigation')}, 'Admin', 'Full system administrator'),
        (${deptMap.get('Litigation')}, 'Manager', 'Department manager with elevated privileges'),
        (${deptMap.get('Litigation')}, 'Clerk', 'Administrative clerk'),
        (${deptMap.get('Litigation')}, 'Viewer', 'Read-only access'),
        (${deptMap.get('Corporate')}, 'Senior Lawyer', 'Senior corporate lawyer'),
        (${deptMap.get('Corporate')}, 'Associate', 'Associate lawyer'),
        (${deptMap.get('Family Law')}, 'Family Lawyer', 'Family law specialist'),
        (${deptMap.get('Family Law')}, 'Mediator', 'Family mediation specialist')
      ON DUPLICATE KEY UPDATE id=id`);
        logger.info('✓ Roles seeded');
        // Get role IDs
        const [roles] = await pool.query('SELECT id, name, department_id FROM roles ORDER BY id');
        const adminRole = roles.find(r => r.name === 'Admin');
        const managerRole = roles.find(r => r.name === 'Manager');
        const viewerRole = roles.find(r => r.name === 'Viewer');
        // ====================
        // Seed Employees
        // ====================
        logger.info('Seeding employees...');
        await pool.query(`INSERT INTO employees (employee_number, first_name, middle_name, surname, dob, phone, email, address, department_id, role_id) VALUES
        ('EMP001', 'John', 'Michael', 'Doe', '1985-05-15', '+1-555-0101', 'john.doe@lawfirm.com', '123 Legal St, Law City', ${deptMap.get('Litigation')}, ${adminRole.id}),
        ('EMP002', 'Jane', NULL, 'Smith', '1990-08-22', '+1-555-0102', 'jane.smith@lawfirm.com', '456 Justice Ave, Law City', ${deptMap.get('Litigation')}, ${managerRole.id}),
        ('EMP003', 'Robert', 'James', 'Johnson', '1988-03-10', '+1-555-0103', 'robert.johnson@lawfirm.com', '789 Court Rd, Law City', ${deptMap.get('Corporate')}, ${roles.find(r => r.name === 'Senior Lawyer')?.id}),
        ('EMP004', 'Emily', 'Grace', 'Williams', '1992-11-05', '+1-555-0104', 'emily.williams@lawfirm.com', '321 Lawyer Ln, Law City', ${deptMap.get('Family Law')}, ${roles.find(r => r.name === 'Family Lawyer')?.id}),
        ('EMP005', 'Michael', NULL, 'Brown', '1987-07-18', '+1-555-0105', 'michael.brown@lawfirm.com', '654 Advocate Blvd, Law City', ${deptMap.get('Litigation')}, ${viewerRole.id}),
        ('EMP006', 'Sarah', 'Ann', 'Davis', '1991-02-14', '+1-555-0106', 'sarah.davis@lawfirm.com', '987 Counsel Ct, Law City', ${deptMap.get('Corporate')}, ${roles.find(r => r.name === 'Associate')?.id}),
        ('EMP007', 'David', 'Lee', 'Martinez', '1989-09-25', '+1-555-0107', 'david.martinez@lawfirm.com', '147 Barrister Way, Law City', ${deptMap.get('Family Law')}, ${roles.find(r => r.name === 'Mediator')?.id}),
        ('EMP008', 'Lisa', NULL, 'Garcia', '1993-04-30', '+1-555-0108', 'lisa.garcia@lawfirm.com', '258 Attorney St, Law City', ${deptMap.get('Litigation')}, ${roles.find(r => r.name === 'Clerk')?.id}),
        ('EMP009', 'James', 'William', 'Rodriguez', '1986-12-08', '+1-555-0109', 'james.rodriguez@lawfirm.com', '369 Solicitor Ave, Law City', ${deptMap.get('Corporate')}, ${roles.find(r => r.name === 'Senior Lawyer')?.id}),
        ('EMP010', 'Patricia', 'Marie', 'Wilson', '1994-06-20', '+1-555-0110', 'patricia.wilson@lawfirm.com', '741 Legal Plaza, Law City', ${deptMap.get('Family Law')}, ${roles.find(r => r.name === 'Family Lawyer')?.id})
      ON DUPLICATE KEY UPDATE id=id`);
        logger.info('✓ Employees seeded');
        // Get employee IDs
        const [employees] = await pool.query('SELECT id, employee_number FROM employees ORDER BY id');
        const empMap = new Map(employees.map(e => [e.employee_number, e.id]));
        // ====================
        // Seed Users
        // ====================
        logger.info('Seeding users...');
        const adminPassword = await hashPassword('password');
        const managerPassword = await hashPassword('password');
        const viewerPassword = await hashPassword('password');
        await pool.query(`INSERT INTO users (employee_id, username, password_hash, department_id, role_id) VALUES
        (${empMap.get('EMP001')}, 'admin', ?, ${deptMap.get('Litigation')}, ${adminRole.id}),
        (${empMap.get('EMP002')}, 'manager', ?, ${deptMap.get('Litigation')}, ${managerRole.id}),
        (${empMap.get('EMP005')}, 'viewer', ?, ${deptMap.get('Litigation')}, ${viewerRole.id})
      ON DUPLICATE KEY UPDATE id=id`, [adminPassword, managerPassword, viewerPassword]);
        logger.info('✓ Users seeded (username: admin, manager, viewer | password: password)');
        // Get IDs for dashboard seeding
        const [users] = await pool.query('SELECT id FROM users LIMIT 1');
        const userId = users[0].id;
        // ====================
        // Seed Lawyers
        // ====================
        logger.info('Seeding lawyers...');
        await pool.query(`INSERT INTO lawyers (employee_id, bar_number, specialization) VALUES
        (${empMap.get('EMP003')}, 'BAR2015001', 'Corporate Law'),
        (${empMap.get('EMP004')}, 'BAR2017002', 'Family Law'),
        (${empMap.get('EMP009')}, 'BAR2014003', 'Corporate Law'),
        (${empMap.get('EMP010')}, 'BAR2018004', 'Family Law')
      ON DUPLICATE KEY UPDATE id=id`);
        logger.info('✓ Lawyers seeded');
        const [lawyers] = await pool.query('SELECT id FROM lawyers');
        // ====================
        // Seed Clients
        // ====================
        logger.info('Seeding clients...');
        await pool.query(`INSERT INTO clients (client_type, name, email, phone, address) VALUES
        ('Corporate', 'Acme Corporation', 'contact@acme.com', '+1-555-1001', '100 Business Park, Metro City'),
        ('Individual', 'Alice Thompson', 'alice.thompson@email.com', '+1-555-1002', '200 Residential St, Suburb'),
        ('Government', 'City Council', 'info@cityhall.gov', '+1-555-1003', '300 Municipal Blvd, Capital City'),
        ('NGO', 'Charity Foundation', 'hello@charity.org', '+1-555-1004', '400 Nonprofit Way, Town'),
        ('Individual', 'Bob Anderson', 'bob.anderson@email.com', '+1-555-1005', '500 Home Ave, Village'),
        ('Corporate', 'Tech Startup Inc', 'support@techstartup.com', '+1-555-1006', '600 Innovation Dr, Tech City')
      ON DUPLICATE KEY UPDATE id=id`);
        logger.info('✓ Clients seeded');
        const [clients] = await pool.query('SELECT id FROM clients');
        // ====================
        // Seed Cases
        // ====================
        logger.info('Seeding cases...');
        const caseValues = [];
        const caseTypes = ['Civil', 'Criminal', 'Corporate', 'Family', 'Property', 'Labour'];
        const caseStatuses = ['Active', 'Pending', 'Closed', 'Won', 'Lost'];
        const locations = [
            { name: 'New York', lat: 40.7128, lng: -74.0060 },
            { name: 'Los Angeles', lat: 34.0522, lng: -118.2437 },
            { name: 'Chicago', lat: 41.8781, lng: -87.6298 },
            { name: 'Houston', lat: 29.7604, lng: -95.3698 },
            { name: 'Phoenix', lat: 33.4484, lng: -112.0740 },
        ];
        for (let i = 1; i <= 50; i++) {
            const type = caseTypes[Math.floor(Math.random() * caseTypes.length)];
            const status = caseStatuses[Math.floor(Math.random() * caseStatuses.length)];
            const client = clients[Math.floor(Math.random() * clients.length)];
            const lawyer = lawyers[Math.floor(Math.random() * lawyers.length)];
            const location = locations[Math.floor(Math.random() * locations.length)];
            const filingDate = new Date(2023, Math.floor(Math.random() * 12), Math.floor(Math.random() * 28) + 1);
            const closingDate = ['Closed', 'Won', 'Lost'].includes(status)
                ? new Date(filingDate.getTime() + Math.random() * 180 * 24 * 60 * 60 * 1000)
                : null;
            caseValues.push(`('CASE${String(i).padStart(4, '0')}', '${type}', '${status}', ${client.id}, ${lawyer.id}, 
          'Case Title ${i}', 'Description for case ${i}', '${filingDate.toISOString().split('T')[0]}', 
          ${closingDate ? `'${closingDate.toISOString().split('T')[0]}'` : 'NULL'}, 
          '${location.name}', ${location.lat}, ${location.lng})`);
        }
        await pool.query(`INSERT INTO cases (case_number, case_type, case_status, client_id, lawyer_id, title, description, 
        filing_date, closing_date, location, latitude, longitude) VALUES ${caseValues.join(',')}
      ON DUPLICATE KEY UPDATE id=id`);
        logger.info('✓ Cases seeded');
        const [cases] = await pool.query('SELECT id FROM cases');
        // ====================
        // Seed Invoices
        // ====================
        logger.info('Seeding invoices...');
        const invoiceValues = [];
        for (let i = 1; i <= 30; i++) {
            const caseItem = cases[Math.floor(Math.random() * cases.length)];
            const client = clients[Math.floor(Math.random() * clients.length)];
            const amount = (Math.random() * 50000 + 5000).toFixed(2);
            const paidAmount = Math.random() > 0.3 ? amount : (parseFloat(amount) * Math.random()).toFixed(2);
            const invoiceDate = new Date(2024, Math.floor(Math.random() * 11), Math.floor(Math.random() * 28) + 1);
            const dueDate = new Date(invoiceDate.getTime() + 30 * 24 * 60 * 60 * 1000);
            const status = parseFloat(paidAmount) >= parseFloat(amount) ? 'Paid' :
                new Date() > dueDate ? 'Overdue' : 'Sent';
            invoiceValues.push(`('INV${String(i).padStart(5, '0')}', ${caseItem.id}, ${client.id}, ${amount}, ${paidAmount}, 
          '${invoiceDate.toISOString().split('T')[0]}', '${dueDate.toISOString().split('T')[0]}', '${status}')`);
        }
        await pool.query(`INSERT INTO invoices (invoice_number, case_id, client_id, amount, paid_amount, invoice_date, due_date, status) 
        VALUES ${invoiceValues.join(',')}
      ON DUPLICATE KEY UPDATE id=id`);
        logger.info('✓ Invoices seeded');
        // ====================
        // Seed Expenses
        // ====================
        logger.info('Seeding expenses...');
        const expenseCategories = ['Court Fees', 'Travel', 'Research', 'Expert Witness', 'Filing Fees', 'Office Supplies'];
        const expenseValues = [];
        for (let i = 1; i <= 40; i++) {
            const caseItem = cases[Math.floor(Math.random() * cases.length)];
            const category = expenseCategories[Math.floor(Math.random() * expenseCategories.length)];
            const amount = (Math.random() * 5000 + 100).toFixed(2);
            const expenseDate = new Date(2024, Math.floor(Math.random() * 11), Math.floor(Math.random() * 28) + 1);
            expenseValues.push(`(${caseItem.id}, '${category}', 'Expense for ${category}', ${amount}, '${expenseDate.toISOString().split('T')[0]}')`);
        }
        await pool.query(`INSERT INTO expenses (case_id, category, description, amount, expense_date) 
        VALUES ${expenseValues.join(',')}
      ON DUPLICATE KEY UPDATE id=id`);
        logger.info('✓ Expenses seeded');
        // ====================
        // Seed Hearings
        // ====================
        logger.info('Seeding hearings...');
        const hearingValues = [];
        for (let i = 1; i <= 25; i++) {
            const caseItem = cases[Math.floor(Math.random() * cases.length)];
            const hearingDate = new Date(2024, 10 + Math.floor(Math.random() * 2), Math.floor(Math.random() * 28) + 1, 9 + Math.floor(Math.random() * 8), 0, 0);
            const location = locations[Math.floor(Math.random() * locations.length)];
            hearingValues.push(`(${caseItem.id}, 'Hearing ${i}', '${hearingDate.toISOString().slice(0, 19).replace('T', ' ')}', 
          '${location.name} Courthouse', 'Notes for hearing ${i}')`);
        }
        await pool.query(`INSERT INTO hearings (case_id, title, hearing_date, location, notes) 
        VALUES ${hearingValues.join(',')}
      ON DUPLICATE KEY UPDATE id=id`);
        logger.info('✓ Hearings seeded');
        // ====================
        // Seed Documents
        // ====================
        logger.info('Seeding documents...');
        const docTypes = ['Contract', 'Motion', 'Brief', 'Evidence', 'Correspondence', 'Order'];
        const docValues = [];
        for (let i = 1; i <= 35; i++) {
            const caseItem = cases[Math.floor(Math.random() * cases.length)];
            const docType = docTypes[Math.floor(Math.random() * docTypes.length)];
            const fileSize = Math.floor(Math.random() * 5000000 + 50000);
            docValues.push(`(${caseItem.id}, '${docType}', 'document_${i}.pdf', '/uploads/documents/document_${i}.pdf', 
          ${fileSize}, ${userId})`);
        }
        await pool.query(`INSERT INTO documents (case_id, document_type, file_name, file_path, file_size, uploaded_by) 
        VALUES ${docValues.join(',')}
      ON DUPLICATE KEY UPDATE id=id`);
        logger.info('✓ Documents seeded');
        // ====================
        // Seed Feedback
        // ====================
        logger.info('Seeding feedback...');
        const feedbackValues = [];
        for (let i = 1; i <= 20; i++) {
            const caseItem = cases[Math.floor(Math.random() * cases.length)];
            const client = clients[Math.floor(Math.random() * clients.length)];
            const rating = Math.floor(Math.random() * 5) + 1;
            const comments = [
                'Excellent service and professional handling',
                'Very satisfied with the outcome',
                'Could improve communication',
                'Outstanding legal representation',
                'Met expectations',
            ];
            feedbackValues.push(`(${caseItem.id}, ${client.id}, ${rating}, '${comments[rating - 1]}')`);
        }
        await pool.query(`INSERT INTO feedback (case_id, client_id, rating, comments) 
        VALUES ${feedbackValues.join(',')}
      ON DUPLICATE KEY UPDATE id=id`);
        logger.info('✓ Feedback seeded');
        logger.info('✓ Database seeding completed successfully!');
        logger.info('\nDefault user credentials:');
        logger.info('  Admin    - username: admin    | password: password');
        logger.info('  Manager  - username: manager  | password: password');
        logger.info('  Viewer   - username: viewer   | password: password');
    }
    catch (error) {
        logger.error({ err: error }, 'Seeding failed');
        throw error;
    }
    finally {
        await pool.end();
    }
}
// Run seeder if this file is executed directly
const isMainModule = import.meta.url.endsWith('seed.ts') || import.meta.url.endsWith('seed.js');
if (isMainModule) {
    console.log('Starting database seeding...');
    seed()
        .then(() => {
        console.log('✅ Seeding process completed successfully');
        logger.info('Seeding process completed');
        process.exit(0);
    })
        .catch((error) => {
        console.error('❌ Seeding process failed:', error);
        logger.error({ err: error }, 'Seeding process failed');
        process.exit(1);
    });
}
export { seed };
//# sourceMappingURL=seed.js.map