import { getPool } from '../db/pool.js';
import { AppError } from './errors.js';
export const authenticate = async (req, res, next) => {
    try {
        const sessionId = req.cookies.sessionId;
        if (!sessionId) {
            throw new AppError(401, 'Authentication required');
        }
        const pool = await getPool();
        // Verify session and get user data
        const [rows] = await pool.query(`SELECT 
        s.user_id,
        u.username,
        u.employee_id,
        u.department_id,
        u.role_id,
        u.is_active
      FROM sessions s
      JOIN users u ON u.id = s.user_id
      WHERE s.id = ? AND s.expires_at > NOW() AND u.is_active = 1`, [sessionId]);
        if (rows.length === 0) {
            res.clearCookie('sessionId');
            throw new AppError(401, 'Invalid or expired session');
        }
        const user = rows[0];
        // Update last used timestamp
        await pool.query('UPDATE sessions SET last_used_at = NOW() WHERE id = ?', [sessionId]);
        // Attach user to request
        req.user = {
            id: user.user_id,
            username: user.username,
            employee_id: user.employee_id,
            department_id: user.department_id,
            role_id: user.role_id,
        };
        next();
    }
    catch (error) {
        next(error);
    }
};
export const optionalAuth = async (req, res, next) => {
    try {
        const sessionId = req.cookies.sessionId;
        if (!sessionId) {
            next();
            return;
        }
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        s.user_id,
        u.username,
        u.employee_id,
        u.department_id,
        u.role_id
      FROM sessions s
      JOIN users u ON u.id = s.user_id
      WHERE s.id = ? AND s.expires_at > NOW() AND u.is_active = 1`, [sessionId]);
        if (rows.length > 0) {
            const user = rows[0];
            req.user = {
                id: user.user_id,
                username: user.username,
                employee_id: user.employee_id,
                department_id: user.department_id,
                role_id: user.role_id,
            };
        }
        next();
    }
    catch (error) {
        next(error);
    }
};
//# sourceMappingURL=auth.js.map