import { ZodError } from 'zod';
import { logger } from '../utils/logger.js';
export class AppError extends Error {
    statusCode;
    message;
    errors;
    constructor(statusCode, message, errors) {
        super(message);
        this.statusCode = statusCode;
        this.message = message;
        this.errors = errors;
        this.name = 'AppError';
        Error.captureStackTrace(this, this.constructor);
    }
}
export const errorHandler = (err, req, res, next) => {
    // Log error
    logger.error({
        err,
        requestId: req.id,
        method: req.method,
        url: req.url,
    }, 'Request error');
    // Handle Zod validation errors
    if (err instanceof ZodError) {
        const errors = {};
        err.errors.forEach((error) => {
            const path = error.path.join('.');
            errors[path] = error.message;
        });
        res.status(422).json({
            success: false,
            error: 'Validation failed',
            errors,
        });
        return;
    }
    // Handle custom AppError
    if (err instanceof AppError) {
        res.status(err.statusCode).json({
            success: false,
            error: err.message,
            errors: err.errors,
        });
        return;
    }
    // Handle MySQL errors
    if (err.code) {
        const mysqlError = err;
        // Duplicate entry
        if (mysqlError.code === 'ER_DUP_ENTRY') {
            res.status(409).json({
                success: false,
                error: 'Duplicate entry. A record with this value already exists.',
            });
            return;
        }
        // Foreign key constraint
        if (mysqlError.code === 'ER_NO_REFERENCED_ROW_2') {
            res.status(400).json({
                success: false,
                error: 'Invalid reference. The referenced record does not exist.',
            });
            return;
        }
    }
    // Default error
    res.status(500).json({
        success: false,
        error: 'Internal server error',
        ...(process.env.NODE_ENV === 'development' && { message: err.message }),
    });
};
export const notFoundHandler = (req, res) => {
    res.status(404).json({
        success: false,
        error: 'Resource not found',
    });
};
//# sourceMappingURL=errors.js.map