import { getPool } from '../../db/pool.js';
import { verifyPassword } from '../../utils/crypto.js';
import { generateUUID } from '../../utils/crypto.js';
import { config } from '../../config.js';
export class AuthRepository {
    /**
     * Find user by username with all related data
     */
    async findUserByUsername(username) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        u.id,
        u.username,
        u.password_hash,
        u.employee_id,
        CONCAT(e.first_name, ' ', IFNULL(CONCAT(e.middle_name, ' '), ''), e.surname) as employee_name,
        u.department_id,
        d.name as department_name,
        u.role_id,
        r.name as role_name,
        u.is_active
      FROM users u
      JOIN employees e ON e.id = u.employee_id
      JOIN departments d ON d.id = u.department_id
      JOIN roles r ON r.id = u.role_id
      WHERE u.username = ?`, [username]);
        return rows.length > 0 ? rows[0] : null;
    }
    /**
     * Verify user credentials
     */
    async verifyCredentials(username, password) {
        const user = await this.findUserByUsername(username);
        if (!user || user.is_active !== 1) {
            return null;
        }
        const isValid = await verifyPassword(user.password_hash, password);
        if (!isValid) {
            return null;
        }
        return user;
    }
    /**
     * Create a new session
     */
    async createSession(userId, ip, userAgent) {
        const pool = await getPool();
        const sessionId = generateUUID();
        const expiresAt = new Date(Date.now() + config.SESSION_MAX_AGE);
        await pool.query(`INSERT INTO sessions (id, user_id, created_at, last_used_at, expires_at, ip, user_agent)
       VALUES (?, ?, NOW(), NOW(), ?, ?, ?)`, [sessionId, userId, expiresAt, ip, userAgent]);
        return {
            id: sessionId,
            user_id: userId,
            expires_at: expiresAt,
        };
    }
    /**
     * Get session by ID
     */
    async getSession(sessionId) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT id, user_id, expires_at
       FROM sessions
       WHERE id = ? AND expires_at > NOW()`, [sessionId]);
        return rows.length > 0 ? rows[0] : null;
    }
    /**
     * Delete session (logout)
     */
    async deleteSession(sessionId) {
        const pool = await getPool();
        await pool.query('DELETE FROM sessions WHERE id = ?', [sessionId]);
    }
    /**
     * Delete all expired sessions (cleanup)
     */
    async deleteExpiredSessions() {
        const pool = await getPool();
        const [result] = await pool.query('DELETE FROM sessions WHERE expires_at < NOW()');
        return result.affectedRows || 0;
    }
    /**
     * Log authentication event to audit log
     */
    async logAuthEvent(userId, action, meta) {
        const pool = await getPool();
        await pool.query(`INSERT INTO audit_logs (user_id, entity, entity_id, action, meta)
       VALUES (?, 'auth', ?, ?, ?)`, [userId, userId || 0, action, JSON.stringify(meta)]);
    }
}
//# sourceMappingURL=repo.js.map