import { Router } from 'express';
import { AuthRepository } from './repo.js';
import { loginSchema } from './dto.js';
import { validate } from '../../middleware/validate.js';
import { authenticate } from '../../middleware/auth.js';
import { AppError } from '../../middleware/errors.js';
import { config } from '../../config.js';
const router = Router();
const authRepo = new AuthRepository();
/**
 * POST /api/auth/login
 * Authenticate user and create session
 */
router.post('/login', validate(loginSchema), async (req, res, next) => {
    try {
        const { username, password } = req.body;
        const ip = req.ip || req.socket.remoteAddress || 'unknown';
        const userAgent = req.headers['user-agent'] || 'unknown';
        // Verify credentials
        const user = await authRepo.verifyCredentials(username, password);
        if (!user) {
            // Log failed attempt
            await authRepo.logAuthEvent(null, 'LOGIN', {
                username,
                ip,
                userAgent,
                success: false,
                reason: 'Invalid credentials',
            });
            throw new AppError(401, 'Invalid username or password');
        }
        // Create session
        const session = await authRepo.createSession(user.id, ip, userAgent);
        // Set HTTP-only secure cookie
        res.cookie('sessionId', session.id, {
            httpOnly: true,
            secure: config.NODE_ENV === 'production',
            sameSite: 'lax',
            maxAge: config.SESSION_MAX_AGE,
        });
        // Log successful login
        await authRepo.logAuthEvent(user.id, 'LOGIN', {
            username: user.username,
            ip,
            userAgent,
            success: true,
        });
        // Return user data (without password hash)
        res.json({
            success: true,
            data: {
                user: {
                    id: user.id,
                    username: user.username,
                    employee_id: user.employee_id,
                    employee_name: user.employee_name,
                    department_id: user.department_id,
                    department_name: user.department_name,
                    role_id: user.role_id,
                    role_name: user.role_name,
                },
            },
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/auth/logout
 * End user session
 */
router.post('/logout', authenticate, async (req, res, next) => {
    try {
        const sessionId = req.cookies.sessionId;
        if (sessionId) {
            await authRepo.deleteSession(sessionId);
            // Log logout
            if (req.user) {
                await authRepo.logAuthEvent(req.user.id, 'LOGOUT', {
                    username: req.user.username,
                    sessionId,
                });
            }
        }
        // Clear cookie
        res.clearCookie('sessionId');
        res.json({
            success: true,
            data: {
                message: 'Logged out successfully',
            },
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * GET /api/me
 * Get current authenticated user info
 */
router.get('/me', authenticate, async (req, res, next) => {
    try {
        if (!req.user) {
            throw new AppError(401, 'Not authenticated');
        }
        // Get full user data
        const user = await authRepo.findUserByUsername(req.user.username);
        if (!user) {
            throw new AppError(404, 'User not found');
        }
        res.json({
            success: true,
            data: {
                user: {
                    id: user.id,
                    username: user.username,
                    employee_id: user.employee_id,
                    employee_name: user.employee_name,
                    department_id: user.department_id,
                    department_name: user.department_name,
                    role_id: user.role_id,
                    role_name: user.role_name,
                },
            },
        });
    }
    catch (error) {
        next(error);
    }
});
export default router;
//# sourceMappingURL=routes.js.map