import { getPool } from '../../db/pool.js';
import { ConflictService } from './service.js';
export class ConflictRepository {
    service;
    constructor() {
        this.service = new ConflictService();
    }
    /**
     * Run comprehensive conflict check
     * Searches across all parties in all matters for potential conflicts
     */
    async runConflictCheck(matterId, data, userId) {
        const pool = await getPool();
        const results = [];
        // Search for each query
        for (const query of data.search_queries) {
            const matches = await this.searchConflicts(query, matterId);
            const conflictLevel = this.service.determineConflictLevel(matches);
            results.push({
                query,
                matches,
                conflict_level: conflictLevel,
            });
        }
        // Determine overall conflict level
        const overallLevel = this.service.determineOverallConflictLevel(results);
        const totalMatches = results.reduce((sum, r) => sum + r.matches.length, 0);
        const summary = this.service.generateResultSummary(results);
        // Save conflict check record
        const [result] = await pool.query(`INSERT INTO conflict_checks (
        matter_id, check_query, run_by, result_count, result_summary, conflict_level
      ) VALUES (?, ?, ?, ?, ?, ?)`, [
            matterId,
            data.search_queries.join('; '),
            userId,
            totalMatches,
            summary,
            overallLevel,
        ]);
        return {
            checkId: result.insertId,
            results,
        };
    }
    /**
     * Search for conflicts across all matters and parties
     * Returns matches with similarity scoring
     */
    async searchConflicts(query, excludeMatterId) {
        const pool = await getPool();
        const matches = [];
        const normalizedQuery = this.service.normalizeName(query);
        // Search in conflict_parties table (full-text + fuzzy)
        const [parties] = await pool.query(`SELECT 
        cp.matter_id,
        c.matter_id as matter_number,
        c.case_type,
        cp.name as party_name,
        cp.party_type,
        cp.identifier,
        MATCH(cp.name) AGAINST(? IN NATURAL LANGUAGE MODE) as relevance
       FROM conflict_parties cp
       JOIN cases c ON c.id = cp.matter_id
       WHERE cp.matter_id != ?
         AND c.is_active = 1
       ORDER BY relevance DESC, cp.created_at DESC
       LIMIT 100`, [query, excludeMatterId]);
        // Process parties - exact and fuzzy matches
        for (const party of parties) {
            const normalizedPartyName = this.service.normalizeName(party.party_name);
            const similarity = this.service.calculateSimilarity(normalizedQuery, normalizedPartyName);
            // Exact match (case-insensitive)
            if (normalizedQuery === normalizedPartyName) {
                matches.push({
                    matter_id: party.matter_id,
                    matter_number: party.matter_number || `CASE-${party.matter_id}`,
                    case_type: party.case_type,
                    party_name: party.party_name,
                    party_type: party.party_type,
                    match_type: 'Exact',
                    similarity_score: 1.0,
                    identifier_match: false,
                });
            }
            // Fuzzy match (80%+ similar)
            else if (similarity >= 0.8) {
                matches.push({
                    matter_id: party.matter_id,
                    matter_number: party.matter_number || `CASE-${party.matter_id}`,
                    case_type: party.case_type,
                    party_name: party.party_name,
                    party_type: party.party_type,
                    match_type: 'Fuzzy',
                    similarity_score: similarity,
                    identifier_match: false,
                });
            }
            // Check identifier match (Ghana Card, Passport, etc.)
            if (party.identifier && query.includes(party.identifier)) {
                // Check if already added
                const existingMatch = matches.find((m) => m.matter_id === party.matter_id && m.party_name === party.party_name);
                if (existingMatch) {
                    existingMatch.identifier_match = true;
                }
                else {
                    matches.push({
                        matter_id: party.matter_id,
                        matter_number: party.matter_number || `CASE-${party.matter_id}`,
                        case_type: party.case_type,
                        party_name: party.party_name,
                        party_type: party.party_type,
                        match_type: 'Exact',
                        similarity_score: 1.0,
                        identifier_match: true,
                    });
                }
            }
        }
        // Search in clients table (name match)
        const [clients] = await pool.query(`SELECT DISTINCT
        c.id as matter_id,
        c.matter_id as matter_number,
        c.case_type,
        cl.name as party_name,
        'Client' as party_type
       FROM cases c
       JOIN clients cl ON cl.id = c.client_id
       WHERE c.id != ?
         AND c.is_active = 1
         AND cl.is_active = 1
         AND cl.name LIKE ?
       LIMIT 50`, [excludeMatterId, `%${query}%`]);
        // Process client matches
        for (const client of clients) {
            const normalizedClientName = this.service.normalizeName(client.party_name);
            const similarity = this.service.calculateSimilarity(normalizedQuery, normalizedClientName);
            if (normalizedQuery === normalizedClientName) {
                matches.push({
                    matter_id: client.matter_id,
                    matter_number: client.matter_number || `CASE-${client.matter_id}`,
                    case_type: client.case_type,
                    party_name: client.party_name,
                    party_type: 'Client',
                    match_type: 'Exact',
                    similarity_score: 1.0,
                    identifier_match: false,
                });
            }
            else if (similarity >= 0.8) {
                matches.push({
                    matter_id: client.matter_id,
                    matter_number: client.matter_number || `CASE-${client.matter_id}`,
                    case_type: client.case_type,
                    party_name: client.party_name,
                    party_type: 'Client',
                    match_type: 'Fuzzy',
                    similarity_score: similarity,
                    identifier_match: false,
                });
            }
        }
        // Remove duplicates and sort by relevance
        const uniqueMatches = this.deduplicateMatches(matches);
        return uniqueMatches.sort((a, b) => {
            // Sort by: exact matches first, then by similarity score
            if (a.match_type === 'Exact' && b.match_type !== 'Exact')
                return -1;
            if (a.match_type !== 'Exact' && b.match_type === 'Exact')
                return 1;
            return b.similarity_score - a.similarity_score;
        });
    }
    /**
     * Remove duplicate matches (same matter + party)
     */
    deduplicateMatches(matches) {
        const seen = new Set();
        return matches.filter((match) => {
            const key = `${match.matter_id}-${match.party_name}`;
            if (seen.has(key))
                return false;
            seen.add(key);
            return true;
        });
    }
    /**
     * Get conflict check by ID with full results
     */
    async findById(id) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        cc.id, cc.matter_id, cc.check_query, cc.run_by,
        CONCAT(e1.first_name, ' ', e1.surname) as run_by_name,
        cc.run_at, cc.result_count, cc.result_summary, cc.conflict_level,
        cc.override_flag, cc.override_by,
        CONCAT(e2.first_name, ' ', e2.surname) as override_by_name,
        cc.override_at, cc.override_justification
       FROM conflict_checks cc
       JOIN users u1 ON u1.id = cc.run_by
       JOIN employees e1 ON e1.id = u1.employee_id
       LEFT JOIN users u2 ON u2.id = cc.override_by
       LEFT JOIN employees e2 ON e2.id = u2.employee_id
       WHERE cc.id = ?`, [id]);
        if (rows.length === 0)
            return null;
        const check = rows[0];
        // Re-run search to get results (or retrieve from cache if stored)
        // For now, we'll return the stored summary
        // In production, you might store full results in a JSON column
        return {
            ...check,
            results: [], // Would need to re-run or store results
        };
    }
    /**
     * Get all conflict checks for a matter
     */
    async findByMatterId(matterId) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        cc.id, cc.matter_id, cc.check_query, cc.run_by,
        CONCAT(e1.first_name, ' ', e1.surname) as run_by_name,
        cc.run_at, cc.result_count, cc.result_summary, cc.conflict_level,
        cc.override_flag, cc.override_by,
        CONCAT(e2.first_name, ' ', e2.surname) as override_by_name,
        cc.override_at, cc.override_justification
       FROM conflict_checks cc
       JOIN users u1 ON u1.id = cc.run_by
       JOIN employees e1 ON e1.id = u1.employee_id
       LEFT JOIN users u2 ON u2.id = cc.override_by
       LEFT JOIN employees e2 ON e2.id = u2.employee_id
       WHERE cc.matter_id = ?
       ORDER BY cc.run_at DESC`, [matterId]);
        return rows.map((row) => ({ ...row, results: [] }));
    }
    /**
     * Override a conflict check (partner approval)
     */
    async overrideConflict(checkId, userId, justification) {
        const pool = await getPool();
        const [result] = await pool.query(`UPDATE conflict_checks 
       SET override_flag = 1, override_by = ?, override_at = NOW(), override_justification = ?
       WHERE id = ?`, [userId, justification, checkId]);
        return result.affectedRows > 0;
    }
}
//# sourceMappingURL=repo.js.map