import { Router } from 'express';
import { ConflictRepository } from './repo.js';
import { runConflictCheckSchema, getConflictCheckSchema, overrideConflictSchema, getMatterConflictsSchema, } from './dto.js';
import { validate } from '../../middleware/validate.js';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { logger } from '../../utils/logger.js';
const router = Router();
const repo = new ConflictRepository();
// All routes require authentication
router.use(authenticate);
/**
 * POST /api/conflicts/check
 * Run a conflicts search
 */
router.post('/check', requirePermission('conflicts:check'), validate(runConflictCheckSchema), async (req, res, next) => {
    try {
        const data = req.body;
        const userId = req.user.id;
        const { checkId, results } = await repo.runConflictCheck(data.matter_id, data, userId);
        // Determine if matter can proceed
        const hasDirectConflict = results.some((r) => r.conflict_level === 'Direct');
        const requiresApproval = results.some((r) => r.conflict_level === 'Indirect' || r.conflict_level === 'Direct');
        // Log audit
        logger.info({
            userId,
            action: 'RUN_CONFLICT_CHECK',
            matterId: data.matter_id,
            checkId,
            conflictLevel: results[0]?.conflict_level || 'None',
        }, 'Conflicts check performed');
        res.status(201).json({
            success: true,
            message: 'Conflicts check completed',
            data: {
                checkId,
                results,
                hasDirectConflict,
                requiresApproval,
                canProceed: !hasDirectConflict,
            },
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to run conflict check');
        next(error);
    }
});
/**
 * GET /api/conflicts/check/:id
 * Get conflict check results by ID
 */
router.get('/check/:id', requirePermission('conflicts:check'), validate(getConflictCheckSchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const check = await repo.findById(parseInt(id));
        if (!check) {
            return res.status(404).json({
                success: false,
                error: 'Conflict check not found',
            });
        }
        res.json({
            success: true,
            data: check,
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, checkId: req.params.id }, 'Failed to get conflict check');
        next(error);
    }
});
/**
 * POST /api/conflicts/:id/override
 * Override a conflict check (requires partner permission)
 */
router.post('/:id/override', requirePermission('conflicts:override'), validate(overrideConflictSchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const { justification } = req.body;
        const userId = req.user.id;
        const overridden = await repo.overrideConflict(parseInt(id), userId, justification);
        if (!overridden) {
            return res.status(404).json({
                success: false,
                error: 'Conflict check not found',
            });
        }
        // Log audit
        logger.info({
            userId,
            action: 'OVERRIDE_CONFLICT',
            checkId: id,
            justification,
        }, 'Conflict check overridden');
        res.json({
            success: true,
            message: 'Conflict override recorded successfully',
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, checkId: req.params.id }, 'Failed to override conflict');
        next(error);
    }
});
/**
 * GET /api/matters/:id/conflicts
 * Get all conflict checks for a matter
 */
router.get('/matters/:id/conflicts', requirePermission('conflicts:check'), validate(getMatterConflictsSchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const checks = await repo.findByMatterId(parseInt(id));
        res.json({
            success: true,
            data: { checks },
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, matterId: req.params.id }, 'Failed to get matter conflicts');
        next(error);
    }
});
export default router;
//# sourceMappingURL=routes.js.map