export class ConflictService {
    /**
     * Calculate Levenshtein distance (edit distance) between two strings
     * Used for fuzzy name matching
     */
    levenshteinDistance(str1, str2) {
        const s1 = str1.toLowerCase();
        const s2 = str2.toLowerCase();
        const costs = [];
        for (let i = 0; i <= s1.length; i++) {
            let lastValue = i;
            for (let j = 0; j <= s2.length; j++) {
                if (i === 0) {
                    costs[j] = j;
                }
                else if (j > 0) {
                    let newValue = costs[j - 1];
                    if (s1.charAt(i - 1) !== s2.charAt(j - 1)) {
                        newValue = Math.min(Math.min(newValue, lastValue), costs[j]) + 1;
                    }
                    costs[j - 1] = lastValue;
                    lastValue = newValue;
                }
            }
            if (i > 0) {
                costs[s2.length] = lastValue;
            }
        }
        return costs[s2.length];
    }
    /**
     * Calculate similarity score (0-1) based on Levenshtein distance
     * 1 = identical, 0 = completely different
     */
    calculateSimilarity(str1, str2) {
        const distance = this.levenshteinDistance(str1, str2);
        const maxLength = Math.max(str1.length, str2.length);
        if (maxLength === 0)
            return 1;
        return 1 - distance / maxLength;
    }
    /**
     * Determine if two names are similar based on similarity threshold
     * Threshold: 0.8 = 80% similar
     */
    isSimilarName(name1, name2, threshold = 0.8) {
        return this.calculateSimilarity(name1, name2) >= threshold;
    }
    /**
     * Determine conflict level based on matches
     * Rules:
     * - Direct: Same client or party appears as adverse party in another matter
     * - Indirect: Related party or counsel appears in another matter
     * - Name Similar: Names are similar but not exact (fuzzy match)
     * - None: No conflicts found
     */
    determineConflictLevel(matches) {
        if (matches.length === 0)
            return 'None';
        // Check for Direct conflicts
        const hasDirectConflict = matches.some((m) => (m.match_type === 'Exact' &&
            (m.party_type === 'Client' || m.party_type === 'Adverse Party')) ||
            m.identifier_match);
        if (hasDirectConflict)
            return 'Direct';
        // Check for Indirect conflicts
        const hasIndirectConflict = matches.some((m) => m.match_type === 'Exact' &&
            (m.party_type === 'Related Party' || m.party_type === 'Counsel'));
        if (hasIndirectConflict)
            return 'Indirect';
        // Check for Name Similar conflicts (fuzzy matches)
        const hasSimilarNames = matches.some((m) => m.match_type === 'Fuzzy');
        if (hasSimilarNames)
            return 'Name Similar';
        return 'None';
    }
    /**
     * Generate result summary text
     */
    generateResultSummary(results) {
        const totalMatches = results.reduce((sum, r) => sum + r.matches.length, 0);
        if (totalMatches === 0) {
            return 'No conflicts detected';
        }
        const directMatches = results.filter((r) => r.conflict_level === 'Direct').length;
        const indirectMatches = results.filter((r) => r.conflict_level === 'Indirect').length;
        const similarMatches = results.filter((r) => r.conflict_level === 'Name Similar').length;
        const parts = [];
        if (directMatches > 0)
            parts.push(`${directMatches} Direct conflict(s)`);
        if (indirectMatches > 0)
            parts.push(`${indirectMatches} Indirect conflict(s)`);
        if (similarMatches > 0)
            parts.push(`${similarMatches} Similar name(s)`);
        return parts.join(', ') + ` (${totalMatches} total match(es))`;
    }
    /**
     * Determine overall conflict level from multiple results
     */
    determineOverallConflictLevel(results) {
        if (results.some((r) => r.conflict_level === 'Direct'))
            return 'Direct';
        if (results.some((r) => r.conflict_level === 'Indirect'))
            return 'Indirect';
        if (results.some((r) => r.conflict_level === 'Name Similar'))
            return 'Name Similar';
        return 'None';
    }
    /**
     * Normalize name for comparison (remove extra spaces, lowercase)
     */
    normalizeName(name) {
        return name.trim().toLowerCase().replace(/\s+/g, ' ');
    }
}
//# sourceMappingURL=service.js.map