import { Router } from 'express';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { getPool } from '../../db/pool.js';
import { logger } from '../../utils/logger.js';
const router = Router();
// All dashboard endpoints require authentication
router.use(authenticate);
/**
 * GET /api/dashboard/kpis
 * Get key performance indicators
 */
router.get('/kpis', requirePermission('dashboard:read'), async (req, res, next) => {
    try {
        const pool = await getPool();
        // Get all KPIs in parallel
        const [totalCasesResult, activeCasesResult, totalClientsResult, totalRevenueResult, pendingInvoicesResult, upcomingHearingsResult, avgRatingResult,] = await Promise.all([
            pool.query('SELECT COUNT(*) as count FROM cases'),
            pool.query('SELECT COUNT(*) as count FROM cases WHERE case_status IN ("Active", "Pending")'),
            pool.query('SELECT COUNT(*) as count FROM clients WHERE is_active = 1'),
            pool.query('SELECT COALESCE(SUM(paid_amount), 0) as total FROM invoices'),
            pool.query('SELECT COUNT(*) as count FROM invoices WHERE status IN ("Sent", "Overdue")'),
            pool.query('SELECT COUNT(*) as count FROM hearings WHERE hearing_date >= NOW()'),
            pool.query('SELECT COALESCE(AVG(rating), 0) as avg FROM feedback'),
        ]);
        const kpis = {
            totalCases: totalCasesResult[0][0].count,
            activeCases: activeCasesResult[0][0].count,
            totalClients: totalClientsResult[0][0].count,
            totalRevenue: parseFloat(totalRevenueResult[0][0].total),
            pendingInvoices: pendingInvoicesResult[0][0].count,
            upcomingHearings: upcomingHearingsResult[0][0].count,
            avgRating: parseFloat(avgRatingResult[0][0].avg).toFixed(1),
        };
        res.json({ success: true, data: kpis });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to fetch dashboard KPIs');
        next(error);
    }
});
/**
 * GET /api/dashboard/case-stats
 * Get case statistics by type and status
 */
router.get('/case-stats', requirePermission('dashboard:read'), async (req, res, next) => {
    try {
        const pool = await getPool();
        const [casesByType, casesByStatus] = await Promise.all([
            pool.query(`
        SELECT case_type as type, COUNT(*) as count 
        FROM cases 
        GROUP BY case_type
        ORDER BY count DESC
      `),
            pool.query(`
        SELECT case_status as status, COUNT(*) as count 
        FROM cases 
        GROUP BY case_status
        ORDER BY count DESC
      `),
        ]);
        res.json({
            success: true,
            data: {
                byType: casesByType[0],
                byStatus: casesByStatus[0],
            },
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to fetch case statistics');
        next(error);
    }
});
/**
 * GET /api/dashboard/revenue-trend
 * Get monthly revenue trend for the current year
 */
router.get('/revenue-trend', requirePermission('dashboard:read'), async (req, res, next) => {
    try {
        const pool = await getPool();
        const [result] = await pool.query(`
      SELECT 
        DATE_FORMAT(invoice_date, '%Y-%m') as month,
        SUM(paid_amount) as revenue
      FROM invoices
      WHERE YEAR(invoice_date) = YEAR(CURDATE())
      GROUP BY DATE_FORMAT(invoice_date, '%Y-%m')
      ORDER BY month ASC
    `);
        res.json({ success: true, data: result });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to fetch revenue trend');
        next(error);
    }
});
/**
 * GET /api/dashboard/hearings
 * Get upcoming hearings
 */
router.get('/hearings', requirePermission('dashboard:read'), async (req, res, next) => {
    try {
        const pool = await getPool();
        const [hearings] = await pool.query(`
      SELECT 
        h.id,
        h.title,
        h.hearing_date,
        h.location,
        h.notes,
        c.case_number,
        c.title as case_title,
        c.case_type
      FROM hearings h
      JOIN cases c ON h.case_id = c.id
      WHERE h.hearing_date >= NOW()
      ORDER BY h.hearing_date ASC
      LIMIT 50
    `);
        res.json({ success: true, data: { hearings } });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to fetch hearings');
        next(error);
    }
});
/**
 * GET /api/dashboard/case-locations
 * Get case locations for map visualization
 */
router.get('/case-locations', requirePermission('dashboard:read'), async (req, res, next) => {
    try {
        const pool = await getPool();
        const [locations] = await pool.query(`
      SELECT 
        id,
        case_number,
        title,
        case_type,
        case_status,
        location,
        latitude,
        longitude
      FROM cases
      WHERE latitude IS NOT NULL 
        AND longitude IS NOT NULL
        AND is_active = 1
      ORDER BY filing_date DESC
      LIMIT 100
    `);
        res.json({ success: true, data: { locations } });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to fetch case locations');
        next(error);
    }
});
/**
 * GET /api/dashboard/recent-activity
 * Get recent activity (cases, invoices, etc.)
 */
router.get('/recent-activity', requirePermission('dashboard:read'), async (req, res, next) => {
    try {
        const pool = await getPool();
        const [recentCases] = await pool.query(`
      SELECT 
        c.id,
        c.case_number,
        c.title,
        c.case_type,
        c.case_status,
        c.filing_date,
        cl.name as client_name
      FROM cases c
      JOIN clients cl ON c.client_id = cl.id
      ORDER BY c.created_at DESC
      LIMIT 10
    `);
        const [recentInvoices] = await pool.query(`
      SELECT 
        i.id,
        i.invoice_number,
        i.amount,
        i.paid_amount,
        i.status,
        i.invoice_date,
        c.case_number,
        cl.name as client_name
      FROM invoices i
      JOIN cases c ON i.case_id = c.id
      JOIN clients cl ON i.client_id = cl.id
      ORDER BY i.created_at DESC
      LIMIT 10
    `);
        res.json({
            success: true,
            data: {
                recentCases,
                recentInvoices,
            },
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to fetch recent activity');
        next(error);
    }
});
export default router;
//# sourceMappingURL=routes.js.map