import { getPool } from '../../db/pool.js';
export class DepartmentRepository {
    /**
     * Get all departments
     */
    async findAll() {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT id, name, description, created_at
       FROM departments
       ORDER BY name ASC`);
        return rows;
    }
    /**
     * Get department by ID
     */
    async findById(id) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT id, name, description, created_at
       FROM departments
       WHERE id = ?`, [id]);
        return rows.length > 0 ? rows[0] : null;
    }
    /**
     * Check if department name exists (excluding specific ID for updates)
     */
    async nameExists(name, excludeId) {
        const pool = await getPool();
        const query = excludeId
            ? 'SELECT COUNT(*) as count FROM departments WHERE name = ? AND id != ?'
            : 'SELECT COUNT(*) as count FROM departments WHERE name = ?';
        const params = excludeId ? [name, excludeId] : [name];
        const [rows] = await pool.query(query, params);
        return rows[0].count > 0;
    }
    /**
     * Create department
     */
    async create(data) {
        const pool = await getPool();
        const [result] = await pool.query(`INSERT INTO departments (name, description)
       VALUES (?, ?)`, [data.name, data.description || null]);
        const newDept = await this.findById(result.insertId);
        if (!newDept) {
            throw new Error('Failed to create department');
        }
        return newDept;
    }
    /**
     * Update department
     */
    async update(id, data) {
        const pool = await getPool();
        const updates = [];
        const values = [];
        if (data.name !== undefined) {
            updates.push('name = ?');
            values.push(data.name);
        }
        if (data.description !== undefined) {
            updates.push('description = ?');
            values.push(data.description);
        }
        if (updates.length === 0) {
            const dept = await this.findById(id);
            if (!dept) {
                throw new Error('Department not found');
            }
            return dept;
        }
        values.push(id);
        await pool.query(`UPDATE departments SET ${updates.join(', ')} WHERE id = ?`, values);
        const updated = await this.findById(id);
        if (!updated) {
            throw new Error('Department not found');
        }
        return updated;
    }
    /**
     * Delete department
     */
    async delete(id) {
        const pool = await getPool();
        await pool.query('DELETE FROM departments WHERE id = ?', [id]);
    }
    /**
     * Check if department has associated roles
     */
    async hasRoles(id) {
        const pool = await getPool();
        const [rows] = await pool.query('SELECT COUNT(*) as count FROM roles WHERE department_id = ?', [id]);
        return rows[0].count > 0;
    }
    /**
     * Log department action to audit log
     */
    async logAction(userId, departmentId, action, meta) {
        const pool = await getPool();
        await pool.query(`INSERT INTO audit_logs (user_id, entity, entity_id, action, meta)
       VALUES (?, 'department', ?, ?, ?)`, [userId, departmentId, action, JSON.stringify(meta)]);
    }
}
//# sourceMappingURL=repo.js.map