import { Router } from 'express';
import { DepartmentRepository } from './repo.js';
import { createDepartmentSchema, updateDepartmentSchema, deleteDepartmentSchema, getDepartmentSchema, } from './dto.js';
import { validate } from '../../middleware/validate.js';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { AppError } from '../../middleware/errors.js';
const router = Router();
const deptRepo = new DepartmentRepository();
// All routes require authentication
router.use(authenticate);
/**
 * GET /api/departments
 * List all departments
 */
router.get('/', async (req, res, next) => {
    try {
        const departments = await deptRepo.findAll();
        res.json({
            success: true,
            data: { departments },
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * GET /api/departments/:id
 * Get department by ID
 */
router.get('/:id', validate(getDepartmentSchema), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const department = await deptRepo.findById(id);
        if (!department) {
            throw new AppError(404, 'Department not found');
        }
        res.json({
            success: true,
            data: { department },
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/departments
 * Create new department (Admin only)
 */
router.post('/', requirePermission('departments:create'), validate(createDepartmentSchema), async (req, res, next) => {
    try {
        const { name, description } = req.body;
        // Check for duplicate name
        const exists = await deptRepo.nameExists(name);
        if (exists) {
            throw new AppError(409, 'Department name already exists');
        }
        const department = await deptRepo.create({ name, description });
        // Log action
        await deptRepo.logAction(req.user.id, department.id, 'CREATE', {
            name: department.name,
            description: department.description,
        });
        res.status(201).json({
            success: true,
            data: { department },
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * PUT /api/departments/:id
 * Update department (Admin only)
 */
router.put('/:id', requirePermission('departments:update'), validate(updateDepartmentSchema), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const { name, description } = req.body;
        // Check if department exists
        const existing = await deptRepo.findById(id);
        if (!existing) {
            throw new AppError(404, 'Department not found');
        }
        // Check for duplicate name (if name is being changed)
        if (name && name !== existing.name) {
            const exists = await deptRepo.nameExists(name, id);
            if (exists) {
                throw new AppError(409, 'Department name already exists');
            }
        }
        const department = await deptRepo.update(id, { name, description });
        // Log action
        await deptRepo.logAction(req.user.id, id, 'UPDATE', {
            old: existing,
            new: department,
        });
        res.json({
            success: true,
            data: { department },
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * DELETE /api/departments/:id
 * Delete department (Admin only)
 */
router.delete('/:id', requirePermission('departments:delete'), validate(deleteDepartmentSchema), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        // Check if department exists
        const department = await deptRepo.findById(id);
        if (!department) {
            throw new AppError(404, 'Department not found');
        }
        // Check if department has roles
        const hasRoles = await deptRepo.hasRoles(id);
        if (hasRoles) {
            throw new AppError(400, 'Cannot delete department with associated roles. Delete roles first.');
        }
        await deptRepo.delete(id);
        // Log action
        await deptRepo.logAction(req.user.id, id, 'DELETE', {
            name: department.name,
        });
        res.json({
            success: true,
            data: { message: 'Department deleted successfully' },
        });
    }
    catch (error) {
        next(error);
    }
});
export default router;
//# sourceMappingURL=routes.js.map