import { getPool } from '../../db/pool.js';
export class EmployeeRepository {
    /**
     * Get all employees with department and role info
     */
    async findAll() {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        e.id, e.employee_number, e.first_name, e.middle_name, e.surname,
        e.dob, e.phone, e.email, e.address,
        e.department_id, d.name as department_name,
        e.role_id, r.name as role_name,
        e.is_active, e.created_at
       FROM employees e
       JOIN departments d ON d.id = e.department_id
       JOIN roles r ON r.id = e.role_id
       ORDER BY e.surname ASC, e.first_name ASC`);
        return rows;
    }
    /**
     * Get employee by ID
     */
    async findById(id) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        e.id, e.employee_number, e.first_name, e.middle_name, e.surname,
        e.dob, e.phone, e.email, e.address,
        e.department_id, d.name as department_name,
        e.role_id, r.name as role_name,
        e.is_active, e.created_at
       FROM employees e
       JOIN departments d ON d.id = e.department_id
       JOIN roles r ON r.id = e.role_id
       WHERE e.id = ?`, [id]);
        return rows.length > 0 ? rows[0] : null;
    }
    /**
     * Check if email exists (excluding specific ID for updates)
     */
    async emailExists(email, excludeId) {
        const pool = await getPool();
        const query = excludeId
            ? 'SELECT COUNT(*) as count FROM employees WHERE email = ? AND id != ?'
            : 'SELECT COUNT(*) as count FROM employees WHERE email = ?';
        const params = excludeId ? [email, excludeId] : [email];
        const [rows] = await pool.query(query, params);
        return rows[0].count > 0;
    }
    /**
     * Check if department exists
     */
    async departmentExists(departmentId) {
        const pool = await getPool();
        const [rows] = await pool.query('SELECT COUNT(*) as count FROM departments WHERE id = ?', [departmentId]);
        return rows[0].count > 0;
    }
    /**
     * Check if role exists and belongs to department
     */
    async roleExistsInDepartment(roleId, departmentId) {
        const pool = await getPool();
        const [rows] = await pool.query('SELECT COUNT(*) as count FROM roles WHERE id = ? AND department_id = ?', [roleId, departmentId]);
        return rows[0].count > 0;
    }
    /**
     * Create employee
     */
    async create(data) {
        const pool = await getPool();
        const [result] = await pool.query(`INSERT INTO employees 
       (employee_number, first_name, middle_name, surname, dob, phone, email, address, department_id, role_id, is_active)
       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`, [
            data.employee_number,
            data.first_name,
            data.middle_name || null,
            data.surname,
            data.dob,
            data.phone,
            data.email,
            data.address,
            data.department_id,
            data.role_id,
            data.is_active !== undefined ? data.is_active : true,
        ]);
        const newEmployee = await this.findById(result.insertId);
        if (!newEmployee) {
            throw new Error('Failed to create employee');
        }
        return newEmployee;
    }
    /**
     * Update employee
     */
    async update(id, data) {
        const pool = await getPool();
        const updates = [];
        const values = [];
        if (data.employee_number !== undefined) {
            updates.push('employee_number = ?');
            values.push(data.employee_number);
        }
        if (data.first_name !== undefined) {
            updates.push('first_name = ?');
            values.push(data.first_name);
        }
        if (data.middle_name !== undefined) {
            updates.push('middle_name = ?');
            values.push(data.middle_name);
        }
        if (data.surname !== undefined) {
            updates.push('surname = ?');
            values.push(data.surname);
        }
        if (data.dob !== undefined) {
            updates.push('dob = ?');
            values.push(data.dob);
        }
        if (data.phone !== undefined) {
            updates.push('phone = ?');
            values.push(data.phone);
        }
        if (data.email !== undefined) {
            updates.push('email = ?');
            values.push(data.email);
        }
        if (data.address !== undefined) {
            updates.push('address = ?');
            values.push(data.address);
        }
        if (data.department_id !== undefined) {
            updates.push('department_id = ?');
            values.push(data.department_id);
        }
        if (data.role_id !== undefined) {
            updates.push('role_id = ?');
            values.push(data.role_id);
        }
        if (data.is_active !== undefined) {
            updates.push('is_active = ?');
            values.push(data.is_active);
        }
        if (updates.length === 0) {
            const employee = await this.findById(id);
            if (!employee) {
                throw new Error('Employee not found');
            }
            return employee;
        }
        values.push(id);
        await pool.query(`UPDATE employees SET ${updates.join(', ')} WHERE id = ?`, values);
        const updated = await this.findById(id);
        if (!updated) {
            throw new Error('Employee not found');
        }
        return updated;
    }
    /**
     * Delete employee
     */
    async delete(id) {
        const pool = await getPool();
        await pool.query('DELETE FROM employees WHERE id = ?', [id]);
    }
    /**
     * Check if employee has associated user
     */
    async hasUser(id) {
        const pool = await getPool();
        const [rows] = await pool.query('SELECT COUNT(*) as count FROM users WHERE employee_id = ?', [id]);
        return rows[0].count > 0;
    }
    /**
     * Log employee action to audit log
     */
    async logAction(userId, employeeId, action, meta) {
        const pool = await getPool();
        await pool.query(`INSERT INTO audit_logs (user_id, entity, entity_id, action, meta)
       VALUES (?, 'employee', ?, ?, ?)`, [userId, employeeId, action, JSON.stringify(meta)]);
    }
}
//# sourceMappingURL=repo.js.map