import { getPool } from '../../db/pool.js';
export class EvidenceRepository {
    /**
     * Get all evidence items
     */
    async findAll(filters) {
        const pool = await getPool();
        let whereConditions = [];
        const params = [];
        if (filters.matter_id) {
            whereConditions.push('ei.matter_id = ?');
            params.push(filters.matter_id);
        }
        if (filters.evidence_type) {
            whereConditions.push('ei.evidence_type = ?');
            params.push(filters.evidence_type);
        }
        if (filters.current_state) {
            whereConditions.push('ei.current_state = ?');
            params.push(filters.current_state);
        }
        if (filters.search) {
            whereConditions.push('(ei.exhibit_label LIKE ? OR MATCH(ei.description) AGAINST(?))');
            const searchPattern = `%${filters.search}%`;
            params.push(searchPattern, filters.search);
        }
        const whereClause = whereConditions.length > 0 ? `WHERE ${whereConditions.join(' AND ')}` : '';
        const [rows] = await pool.query(`SELECT 
        ei.id, ei.matter_id, c.matter_id as matter_number, c.title as matter_title,
        ei.exhibit_label, ei.description, ei.evidence_type,
        ei.file_path, ei.file_hash, ei.file_size_bytes, ei.file_mime_type,
        ei.collection_date, ei.collection_location, ei.collected_by,
        ei.current_state, ei.storage_location, ei.is_sealed, ei.seal_number,
        ei.linked_hearing_id, ei.notes, ei.created_by, ei.created_at, ei.updated_at
       FROM evidence_items ei
       JOIN cases c ON c.id = ei.matter_id
       ${whereClause}
       ORDER BY ei.exhibit_label`, params);
        return rows;
    }
    /**
     * Find evidence by ID with custody history
     */
    async findById(id) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        ei.id, ei.matter_id, c.matter_id as matter_number, c.title as matter_title,
        ei.exhibit_label, ei.description, ei.evidence_type,
        ei.file_path, ei.file_hash, ei.file_size_bytes, ei.file_mime_type,
        ei.collection_date, ei.collection_location, ei.collected_by,
        ei.current_state, ei.storage_location, ei.is_sealed, ei.seal_number,
        ei.linked_hearing_id, ei.notes, ei.created_by, ei.created_at, ei.updated_at
       FROM evidence_items ei
       JOIN cases c ON c.id = ei.matter_id
       WHERE ei.id = ?`, [id]);
        if (rows.length === 0)
            return null;
        const evidence = rows[0];
        const custody_events = await this.getCustodyHistory(id);
        return { ...evidence, custody_events };
    }
    /**
     * Create evidence item
     */
    async create(data, userId) {
        const pool = await getPool();
        const [result] = await pool.query(`INSERT INTO evidence_items (
        matter_id, exhibit_label, description, evidence_type,
        file_path, file_hash, file_size_bytes, file_mime_type,
        collection_date, collection_location, collected_by,
        current_state, storage_location, is_sealed, seal_number,
        linked_hearing_id, notes, created_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Collected', ?, ?, ?, ?, ?, ?)`, [
            data.matter_id,
            data.exhibit_label,
            data.description,
            data.evidence_type,
            data.file_path || null,
            data.file_hash || null,
            data.file_size_bytes || null,
            data.file_mime_type || null,
            data.collection_date || null,
            data.collection_location || null,
            data.collected_by || null,
            data.storage_location || null,
            data.is_sealed ? 1 : 0,
            data.seal_number || null,
            data.linked_hearing_id || null,
            data.notes || null,
            userId,
        ]);
        // Record initial custody event
        await this.recordCustodyEvent(result.insertId, {
            event_type: 'Collected',
            from_person: undefined,
            to_person: data.collected_by || undefined,
            location: data.collection_location || undefined,
            purpose: 'Initial collection',
            notes: 'Evidence collected and registered in system',
        }, userId);
        return result.insertId;
    }
    /**
     * Update evidence item
     */
    async update(id, data, userId) {
        const pool = await getPool();
        const fields = [];
        const values = [];
        let stateChanged = false;
        let oldState = null;
        let newState = null;
        // Get current state before update
        if (data.current_state) {
            const [current] = await pool.query(`SELECT current_state FROM evidence_items WHERE id = ?`, [id]);
            if (current.length > 0) {
                oldState = current[0].current_state;
                newState = data.current_state;
                stateChanged = oldState !== newState;
            }
        }
        if (data.exhibit_label !== undefined) {
            fields.push('exhibit_label = ?');
            values.push(data.exhibit_label);
        }
        if (data.description !== undefined) {
            fields.push('description = ?');
            values.push(data.description);
        }
        if (data.evidence_type !== undefined) {
            fields.push('evidence_type = ?');
            values.push(data.evidence_type);
        }
        if (data.file_path !== undefined) {
            fields.push('file_path = ?');
            values.push(data.file_path);
        }
        if (data.file_hash !== undefined) {
            fields.push('file_hash = ?');
            values.push(data.file_hash);
        }
        if (data.file_size_bytes !== undefined) {
            fields.push('file_size_bytes = ?');
            values.push(data.file_size_bytes);
        }
        if (data.file_mime_type !== undefined) {
            fields.push('file_mime_type = ?');
            values.push(data.file_mime_type);
        }
        if (data.collection_date !== undefined) {
            fields.push('collection_date = ?');
            values.push(data.collection_date);
        }
        if (data.collection_location !== undefined) {
            fields.push('collection_location = ?');
            values.push(data.collection_location);
        }
        if (data.collected_by !== undefined) {
            fields.push('collected_by = ?');
            values.push(data.collected_by);
        }
        if (data.current_state !== undefined) {
            fields.push('current_state = ?');
            values.push(data.current_state);
        }
        if (data.storage_location !== undefined) {
            fields.push('storage_location = ?');
            values.push(data.storage_location);
        }
        if (data.is_sealed !== undefined) {
            fields.push('is_sealed = ?');
            values.push(data.is_sealed ? 1 : 0);
        }
        if (data.seal_number !== undefined) {
            fields.push('seal_number = ?');
            values.push(data.seal_number);
        }
        if (data.linked_hearing_id !== undefined) {
            fields.push('linked_hearing_id = ?');
            values.push(data.linked_hearing_id);
        }
        if (data.notes !== undefined) {
            fields.push('notes = ?');
            values.push(data.notes);
        }
        if (fields.length === 0)
            return false;
        values.push(id);
        const [result] = await pool.query(`UPDATE evidence_items SET ${fields.join(', ')} WHERE id = ?`, values);
        // Record custody event if state changed
        if (stateChanged && oldState && newState) {
            await this.recordCustodyEvent(id, {
                event_type: newState,
                notes: `State changed from ${oldState} to ${newState}`,
            }, userId);
        }
        return result.affectedRows > 0;
    }
    /**
     * Delete evidence item
     */
    async delete(id) {
        const pool = await getPool();
        const [result] = await pool.query(`DELETE FROM evidence_items WHERE id = ?`, [id]);
        return result.affectedRows > 0;
    }
    /**
     * Record custody event (chain-of-custody)
     */
    async recordCustodyEvent(evidenceId, data, userId) {
        const pool = await getPool();
        const [result] = await pool.query(`INSERT INTO custody_events (
        evidence_id, event_type, from_person, to_person,
        location, purpose, witness_name, notes, recorded_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`, [
            evidenceId,
            data.event_type,
            data.from_person || null,
            data.to_person || null,
            data.location || null,
            data.purpose || null,
            data.witness_name || null,
            data.notes || null,
            userId,
        ]);
        return result.insertId;
    }
    /**
     * Get custody history for evidence
     */
    async getCustodyHistory(evidenceId) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        ce.id, ce.evidence_id, ce.event_type, ce.event_date,
        ce.from_person, ce.to_person, ce.location, ce.purpose,
        ce.witness_name, ce.notes, ce.recorded_by,
        CONCAT(e.first_name, ' ', e.surname) as recorded_by_name,
        ce.recorded_at
       FROM custody_events ce
       JOIN users u ON u.id = ce.recorded_by
       JOIN employees e ON e.id = u.employee_id
       WHERE ce.evidence_id = ?
       ORDER BY ce.event_date DESC, ce.recorded_at DESC`, [evidenceId]);
        return rows;
    }
}
//# sourceMappingURL=repo.js.map