import { Router } from 'express';
import { EvidenceRepository } from './repo.js';
import { createEvidenceSchema, updateEvidenceSchema, getEvidenceSchema, listEvidenceSchema, recordCustodyEventSchema, getCustodyHistorySchema, } from './dto.js';
import { validate } from '../../middleware/validate.js';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { logger } from '../../utils/logger.js';
const router = Router();
const repo = new EvidenceRepository();
// All routes require authentication
router.use(authenticate);
/**
 * GET /api/evidence
 * List evidence items with filters
 */
router.get('/', requirePermission('evidence:read'), validate(listEvidenceSchema), async (req, res, next) => {
    try {
        const { matter_id, evidence_type, current_state, search } = req.query;
        const evidence = await repo.findAll({
            matter_id: matter_id ? parseInt(matter_id) : undefined,
            evidence_type: evidence_type,
            current_state: current_state,
            search: search,
        });
        res.json({
            success: true,
            data: { evidence },
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to list evidence');
        next(error);
    }
});
/**
 * GET /api/evidence/:id
 * Get evidence by ID with custody history
 */
router.get('/:id', requirePermission('evidence:read'), validate(getEvidenceSchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const evidence = await repo.findById(parseInt(id));
        if (!evidence) {
            return res.status(404).json({
                success: false,
                error: 'Evidence not found',
            });
        }
        res.json({
            success: true,
            data: evidence,
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, evidenceId: req.params.id }, 'Failed to get evidence');
        next(error);
    }
});
/**
 * POST /api/evidence
 * Create evidence item
 */
router.post('/', requirePermission('evidence:create'), validate(createEvidenceSchema), async (req, res, next) => {
    try {
        const data = req.body;
        const userId = req.user.id;
        const evidenceId = await repo.create(data, userId);
        // Log audit
        logger.info({
            userId,
            action: 'CREATE_EVIDENCE',
            evidenceId,
            matterId: data.matter_id,
            exhibitLabel: data.exhibit_label,
        }, 'Evidence created');
        const evidence = await repo.findById(evidenceId);
        res.status(201).json({
            success: true,
            message: 'Evidence created successfully',
            data: evidence,
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to create evidence');
        next(error);
    }
});
/**
 * PUT /api/evidence/:id
 * Update evidence item
 */
router.put('/:id', requirePermission('evidence:update'), validate(updateEvidenceSchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const data = req.body;
        const userId = req.user.id;
        const updated = await repo.update(parseInt(id), data, userId);
        if (!updated) {
            return res.status(404).json({
                success: false,
                error: 'Evidence not found or no changes made',
            });
        }
        // Log audit
        logger.info({
            userId,
            action: 'UPDATE_EVIDENCE',
            evidenceId: id,
            changes: data,
        }, 'Evidence updated');
        const evidence = await repo.findById(parseInt(id));
        res.json({
            success: true,
            message: 'Evidence updated successfully',
            data: evidence,
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, evidenceId: req.params.id }, 'Failed to update evidence');
        next(error);
    }
});
/**
 * DELETE /api/evidence/:id
 * Delete evidence item
 */
router.delete('/:id', requirePermission('evidence:delete'), validate(getEvidenceSchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const deleted = await repo.delete(parseInt(id));
        if (!deleted) {
            return res.status(404).json({
                success: false,
                error: 'Evidence not found',
            });
        }
        // Log audit
        logger.info({
            userId: req.user.id,
            action: 'DELETE_EVIDENCE',
            evidenceId: id,
        }, 'Evidence deleted');
        res.json({
            success: true,
            message: 'Evidence deleted successfully',
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, evidenceId: req.params.id }, 'Failed to delete evidence');
        next(error);
    }
});
/**
 * POST /api/evidence/:id/custody
 * Record custody event (chain-of-custody)
 */
router.post('/:id/custody', requirePermission('evidence:custody'), validate(recordCustodyEventSchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const data = req.body;
        const userId = req.user.id;
        // Check if evidence exists
        const evidence = await repo.findById(parseInt(id));
        if (!evidence) {
            return res.status(404).json({
                success: false,
                error: 'Evidence not found',
            });
        }
        const eventId = await repo.recordCustodyEvent(parseInt(id), data, userId);
        // Log audit
        logger.info({
            userId,
            action: 'RECORD_CUSTODY_EVENT',
            evidenceId: id,
            eventType: data.event_type,
        }, 'Custody event recorded');
        res.status(201).json({
            success: true,
            message: 'Custody event recorded successfully',
            data: { id: eventId },
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, evidenceId: req.params.id }, 'Failed to record custody event');
        next(error);
    }
});
/**
 * GET /api/evidence/:id/custody
 * Get custody history for evidence
 */
router.get('/:id/custody', requirePermission('evidence:read'), validate(getCustodyHistorySchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const custodyHistory = await repo.getCustodyHistory(parseInt(id));
        res.json({
            success: true,
            data: { custodyHistory },
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, evidenceId: req.params.id }, 'Failed to get custody history');
        next(error);
    }
});
export default router;
//# sourceMappingURL=routes.js.map