import { Router } from 'express';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { expenseRepo } from './repo.js';
import { createExpenseSchema, updateExpenseSchema, queryExpensesSchema, submitExpenseSchema, approveExpenseSchema, rejectExpenseSchema, createExpenseCategorySchema, updateExpenseCategorySchema, } from './dto.js';
const router = Router();
// ================================================
// Expense Entries
// ================================================
/**
 * GET /api/expenses
 * List expenses with filters
 */
router.get('/', authenticate, requirePermission('expense:create'), async (req, res, next) => {
    try {
        const query = queryExpensesSchema.parse(req.query);
        const canViewAll = false; // Will implement role-based check later
        const result = await expenseRepo.listExpenses(query, req.user.id, canViewAll);
        res.json({ success: true, data: result });
    }
    catch (error) {
        next(error);
    }
});
/**
 * GET /api/expenses/:id
 * Get single expense by ID
 */
router.get('/:id', authenticate, requirePermission('expense:create'), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const canViewAll = false; // Will implement role-based check later
        const expense = await expenseRepo.getExpenseById(id, req.user.id, canViewAll);
        if (!expense) {
            res.status(404).json({ success: false, error: 'Expense not found' });
            return;
        }
        res.json({ success: true, data: expense });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/expenses
 * Create new expense
 */
router.post('/', authenticate, requirePermission('expense:create'), async (req, res, next) => {
    try {
        const data = createExpenseSchema.parse(req.body);
        const expense = await expenseRepo.createExpense(data, req.user.id);
        res.status(201).json({ success: true, data: expense });
    }
    catch (error) {
        next(error);
    }
});
/**
 * PUT /api/expenses/:id
 * Update expense (only Draft or Rejected expenses)
 */
router.put('/:id', authenticate, requirePermission('expense:create'), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const data = updateExpenseSchema.parse(req.body);
        const canViewAll = false; // Will implement role-based check later
        const expense = await expenseRepo.updateExpense(id, data, req.user.id, canViewAll);
        res.json({ success: true, data: expense });
    }
    catch (error) {
        next(error);
    }
});
/**
 * DELETE /api/expenses/:id
 * Delete expense (only Draft expenses)
 */
router.delete('/:id', authenticate, requirePermission('expense:create'), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const canViewAll = false; // Will implement role-based check later
        await expenseRepo.deleteExpense(id, req.user.id, canViewAll);
        res.json({ success: true, message: 'Expense deleted successfully' });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/expenses/submit
 * Submit expenses for approval
 */
router.post('/submit', authenticate, requirePermission('expense:create'), async (req, res, next) => {
    try {
        const { expense_ids } = submitExpenseSchema.parse(req.body);
        await expenseRepo.submitExpenses(expense_ids, req.user.id);
        res.json({ success: true, message: `${expense_ids.length} expenses submitted for approval` });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/expenses/approve
 * Approve expenses (managers/admins only)
 */
router.post('/approve', authenticate, requirePermission('expense:approve'), async (req, res, next) => {
    try {
        const { expense_ids } = approveExpenseSchema.parse(req.body);
        await expenseRepo.approveExpenses(expense_ids, req.user.id);
        res.json({ success: true, message: `${expense_ids.length} expenses approved` });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/expenses/reject
 * Reject expenses (managers/admins only)
 */
router.post('/reject', authenticate, requirePermission('expense:approve'), async (req, res, next) => {
    try {
        const { expense_ids, rejection_reason } = rejectExpenseSchema.parse(req.body);
        await expenseRepo.rejectExpenses(expense_ids, req.user.id, rejection_reason);
        res.json({ success: true, message: `${expense_ids.length} expenses rejected` });
    }
    catch (error) {
        next(error);
    }
});
// ================================================
// Expense Categories
// ================================================
/**
 * GET /api/expenses/categories
 * List expense categories
 */
router.get('/categories', authenticate, async (req, res, next) => {
    try {
        const activeOnly = req.query.active_only !== 'false';
        const categories = await expenseRepo.listExpenseCategories(activeOnly);
        res.json({ success: true, data: { categories } });
    }
    catch (error) {
        next(error);
    }
});
/**
 * GET /api/expenses/categories/:id
 * Get single expense category
 */
router.get('/categories/:id', authenticate, async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const category = await expenseRepo.getExpenseCategoryById(id);
        if (!category) {
            res.status(404).json({ success: false, error: 'Expense category not found' });
            return;
        }
        res.json({ success: true, data: category });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/expenses/categories
 * Create expense category (admin only)
 */
router.post('/categories', authenticate, requirePermission('expense:approve'), // Only managers/admins
async (req, res, next) => {
    try {
        const data = createExpenseCategorySchema.parse(req.body);
        const category = await expenseRepo.createExpenseCategory(data);
        res.status(201).json({ success: true, data: category });
    }
    catch (error) {
        next(error);
    }
});
/**
 * PUT /api/expenses/categories/:id
 * Update expense category (admin only)
 */
router.put('/categories/:id', authenticate, requirePermission('expense:approve'), // Only managers/admins
async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const data = updateExpenseCategorySchema.parse(req.body);
        const category = await expenseRepo.updateExpenseCategory(id, data);
        res.json({ success: true, data: category });
    }
    catch (error) {
        next(error);
    }
});
export default router;
//# sourceMappingURL=routes.js.map